import { LoggingWinston } from "@google-cloud/logging-winston";
import { getCurrentEnv } from "genkit";
import { logger } from "genkit/logging";
import { loggingDenied, loggingDeniedHelpText } from "./utils.js";
let additionalStream;
let useJsonFormatOverride = false;
class GcpLogger {
  constructor(config) {
    this.config = config;
  }
  async getLogger(env) {
    const winston = await import("winston");
    const format = useJsonFormatOverride || this.shouldExport(env) ? { format: winston.format.json() } : {
      format: winston.format.printf((info) => {
        return `[${info.level}] ${info.message}`;
      })
    };
    const transports = [];
    transports.push(
      this.shouldExport(env) ? new LoggingWinston({
        labels: { module: "genkit" },
        prefix: "genkit",
        logName: "genkit_log",
        projectId: this.config.projectId,
        credentials: this.config.credentials,
        autoRetry: true,
        defaultCallback: await this.getErrorHandler()
      }) : new winston.transports.Console()
    );
    if (additionalStream) {
      transports.push(
        new winston.transports.Stream({ stream: additionalStream })
      );
    }
    return winston.createLogger({
      transports,
      ...format,
      exceptionHandlers: [new winston.transports.Console()]
    });
  }
  async getErrorHandler() {
    let instructionsLogged = false;
    const helpInstructions = await loggingDeniedHelpText();
    return async (err) => {
      const defaultLogger = logger.defaultLogger;
      if (err && loggingDenied(err)) {
        if (!instructionsLogged) {
          instructionsLogged = true;
          defaultLogger.error(
            `Unable to send logs to Google Cloud: ${err.message}

${helpInstructions}
`
          );
        }
      } else if (err) {
        defaultLogger.error(`Unable to send logs to Google Cloud: ${err}`);
      }
      if (err) {
        logger.init(
          await new GcpLogger(this.config).getLogger(getCurrentEnv())
        );
        defaultLogger.info("Initialized a new GcpLogger.");
      }
    };
  }
  shouldExport(env) {
    return this.config.export;
  }
}
function __addTransportStreamForTesting(stream) {
  additionalStream = stream;
}
function __useJsonFormatForTesting() {
  useJsonFormatOverride = true;
}
export {
  GcpLogger,
  __addTransportStreamForTesting,
  __useJsonFormatForTesting
};
//# sourceMappingURL=gcpLogger.mjs.map