import { ValueType } from "@opentelemetry/api";
import { GENKIT_VERSION } from "genkit";
import { logger } from "genkit/logging";
import {
  MetricCounter,
  internalMetricNamespaceWrap
} from "../metrics.js";
import { createCommonLogAttributes, truncate } from "../utils.js";
class EngagementTelemetry {
  /**
   * Wraps the declared metrics in a Genkit-specific, internal namespace.
   */
  _N = internalMetricNamespaceWrap.bind(null, "engagement");
  feedbackCounter = new MetricCounter(this._N("feedback"), {
    description: "Counts calls to genkit flows.",
    valueType: ValueType.INT
  });
  acceptanceCounter = new MetricCounter(this._N("acceptance"), {
    description: "Tracks unique flow paths per flow.",
    valueType: ValueType.INT
  });
  tick(span, logInputAndOutput, projectId) {
    const subtype = span.attributes["genkit:metadata:subtype"];
    if (subtype === "userFeedback") {
      this.writeUserFeedback(span, projectId);
      return;
    }
    if (subtype === "userAcceptance") {
      this.writeUserAcceptance(span, projectId);
      return;
    }
    logger.warn(`Unknown user engagement subtype: ${subtype}`);
  }
  writeUserFeedback(span, projectId) {
    const attributes = span.attributes;
    const name = this.extractTraceName(attributes);
    const dimensions = {
      name,
      value: attributes["genkit:metadata:feedbackValue"],
      hasText: !!attributes["genkit:metadata:textFeedback"],
      source: "ts",
      sourceVersion: GENKIT_VERSION
    };
    this.feedbackCounter.add(1, dimensions);
    const metadata = {
      ...createCommonLogAttributes(span, projectId),
      feedbackValue: attributes["genkit:metadata:feedbackValue"]
    };
    if (attributes["genkit:metadata:textFeedback"]) {
      metadata["textFeedback"] = truncate(
        attributes["genkit:metadata:textFeedback"]
      );
    }
    logger.logStructured(`UserFeedback[${name}]`, metadata);
  }
  writeUserAcceptance(span, projectId) {
    const attributes = span.attributes;
    const name = this.extractTraceName(attributes);
    const dimensions = {
      name,
      value: attributes["genkit:metadata:acceptanceValue"],
      source: "ts",
      sourceVersion: GENKIT_VERSION
    };
    this.acceptanceCounter.add(1, dimensions);
    const metadata = {
      ...createCommonLogAttributes(span, projectId),
      acceptanceValue: attributes["genkit:metadata:acceptanceValue"]
    };
    logger.logStructured(`UserAcceptance[${name}]`, metadata);
  }
  extractTraceName(attributes) {
    const path = attributes["genkit:path"];
    if (!path || path === "<unknown>") {
      return "<unknown>";
    }
    const name = path.match("/{(.+)}+");
    return name ? name[1] : "<unknown>";
  }
}
const engagementTelemetry = new EngagementTelemetry();
export {
  engagementTelemetry
};
//# sourceMappingURL=engagement.mjs.map