"use strict";
var __defProp = Object.defineProperty;
var __getOwnPropDesc = Object.getOwnPropertyDescriptor;
var __getOwnPropNames = Object.getOwnPropertyNames;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};
var __copyProps = (to, from, except, desc) => {
  if (from && typeof from === "object" || typeof from === "function") {
    for (let key of __getOwnPropNames(from))
      if (!__hasOwnProp.call(to, key) && key !== except)
        __defProp(to, key, { get: () => from[key], enumerable: !(desc = __getOwnPropDesc(from, key)) || desc.enumerable });
  }
  return to;
};
var __toCommonJS = (mod) => __copyProps(__defProp({}, "__esModule", { value: true }), mod);
var converters_exports = {};
__export(converters_exports, {
  fromGeminiCandidate: () => fromGeminiCandidate,
  toGeminiFunctionModeEnum: () => toGeminiFunctionModeEnum,
  toGeminiMessage: () => toGeminiMessage,
  toGeminiSystemInstruction: () => toGeminiSystemInstruction,
  toGeminiTool: () => toGeminiTool
});
module.exports = __toCommonJS(converters_exports);
var import_genkit = require("genkit");
var import_types = require("./types.js");
function toGeminiTool(tool) {
  const declaration = {
    name: tool.name.replace(/\//g, "__"),
    // Gemini throws on '/' in tool name
    description: tool.description,
    parameters: toGeminiSchemaProperty(tool.inputSchema)
  };
  return declaration;
}
function toGeminiSchemaProperty(property) {
  if (!property || !property.type) {
    return void 0;
  }
  const baseSchema = {};
  if (property.description) {
    baseSchema.description = property.description;
  }
  if (property.enum) {
    baseSchema.enum = property.enum;
  }
  if (property.nullable) {
    baseSchema.nullable = property.nullable;
  }
  let propertyType;
  if (Array.isArray(property.type)) {
    const types = property.type;
    if (types.includes("null")) {
      baseSchema.nullable = true;
    }
    propertyType = types.find((t) => t !== "null");
  } else {
    propertyType = property.type;
  }
  if (propertyType === "object") {
    const nestedProperties = {};
    Object.keys(property.properties).forEach((key) => {
      nestedProperties[key] = toGeminiSchemaProperty(property.properties[key]);
    });
    return {
      ...baseSchema,
      type: import_types.SchemaType.OBJECT,
      properties: nestedProperties,
      required: property.required
    };
  } else if (propertyType === "array") {
    return {
      ...baseSchema,
      type: import_types.SchemaType.ARRAY,
      items: toGeminiSchemaProperty(property.items)
    };
  } else {
    const schemaType = import_types.SchemaType[propertyType.toUpperCase()];
    if (!schemaType) {
      throw new import_genkit.GenkitError({
        status: "INVALID_ARGUMENT",
        message: `Unsupported property type ${propertyType.toUpperCase()}`
      });
    }
    return {
      ...baseSchema,
      type: schemaType
    };
  }
}
function toGeminiMedia(part) {
  if (part.media?.url.startsWith("data:")) {
    const dataUrl = part.media.url;
    const b64Data = dataUrl.substring(dataUrl.indexOf(",") + 1);
    const contentType = part.media.contentType || dataUrl.substring(dataUrl.indexOf(":") + 1, dataUrl.indexOf(";"));
    return { inlineData: { mimeType: contentType, data: b64Data } };
  }
  if (!part.media?.contentType) {
    throw Error(
      "Must supply a `contentType` when sending File URIs to Gemini."
    );
  }
  return {
    fileData: {
      mimeType: part.media.contentType,
      fileUri: part.media.url
    }
  };
}
function toGeminiToolRequest(part) {
  if (!part.toolRequest?.input) {
    throw Error("Invalid ToolRequestPart: input was missing.");
  }
  return {
    functionCall: {
      name: part.toolRequest.name,
      args: part.toolRequest.input
    }
  };
}
function toGeminiToolResponse(part) {
  if (!part.toolResponse?.output) {
    throw Error("Invalid ToolResponsePart: output was missing.");
  }
  return {
    functionResponse: {
      name: part.toolResponse.name,
      response: {
        name: part.toolResponse.name,
        content: part.toolResponse.output
      }
    }
  };
}
function toGeminiReasoning(part) {
  const out = { thought: true };
  if (typeof part.metadata?.thoughtSignature === "string") {
    out.thoughtSignature = part.metadata.thoughtSignature;
  }
  if (part.reasoning?.length) {
    out.text = part.reasoning;
  }
  return out;
}
function toGeminiCustom(part) {
  if (part.custom?.codeExecutionResult) {
    return {
      codeExecutionResult: part.custom.codeExecutionResult
    };
  }
  if (part.custom?.executableCode) {
    return {
      executableCode: part.custom.executableCode
    };
  }
  throw new Error("Unsupported Custom Part type");
}
function toGeminiPart(part) {
  if (part.text) {
    return { text: part.text };
  }
  if (part.media) {
    return toGeminiMedia(part);
  }
  if (part.toolRequest) {
    return toGeminiToolRequest(part);
  }
  if (part.toolResponse) {
    return toGeminiToolResponse(part);
  }
  if (typeof part.reasoning === "string") {
    return toGeminiReasoning(part);
  }
  if (part.custom) {
    return toGeminiCustom(part);
  }
  throw new Error("Unsupported Part type " + JSON.stringify(part));
}
function toGeminiRole(role, model) {
  switch (role) {
    case "user":
      return "user";
    case "model":
      return "model";
    case "system":
      if (model?.info?.supports?.systemRole) {
        throw new Error(
          "system role is only supported for a single message in the first position"
        );
      } else {
        throw new Error("system role is not supported");
      }
    case "tool":
      return "function";
    default:
      return "user";
  }
}
function toGeminiMessage(message, model) {
  let sortedParts = message.content;
  if (message.role === "tool") {
    sortedParts = [...message.content].sort((a, b) => {
      const aRef = a.toolResponse?.ref;
      const bRef = b.toolResponse?.ref;
      if (!aRef && !bRef) return 0;
      if (!aRef) return 1;
      if (!bRef) return -1;
      return parseInt(aRef, 10) - parseInt(bRef, 10);
    });
  }
  return {
    role: toGeminiRole(message.role, model),
    parts: sortedParts.map(toGeminiPart)
  };
}
function toGeminiSystemInstruction(message) {
  return {
    role: "user",
    parts: message.content.map(toGeminiPart)
  };
}
function toGeminiFunctionModeEnum(from) {
  if (from === void 0) {
    return void 0;
  }
  switch (from) {
    case "MODE_UNSPECIFIED": {
      return import_types.FunctionCallingMode.MODE_UNSPECIFIED;
    }
    case "required":
    case "ANY": {
      return import_types.FunctionCallingMode.ANY;
    }
    case "auto":
    case "AUTO": {
      return import_types.FunctionCallingMode.AUTO;
    }
    case "none":
    case "NONE": {
      return import_types.FunctionCallingMode.NONE;
    }
    default:
      throw new Error(`unsupported function calling mode: ${from}`);
  }
}
function fromGeminiFinishReason(reason) {
  if (!reason) return "unknown";
  switch (reason) {
    case "STOP":
      return "stop";
    case "MAX_TOKENS":
      return "length";
    case "SAFETY":
    // blocked for safety
    case "RECITATION":
    // blocked for reciting training data
    case "LANGUAGE":
    // blocked for using an unsupported language
    case "BLOCKLIST":
    // blocked for forbidden terms
    case "PROHIBITED_CONTENT":
    // blocked for potentially containing prohibited content
    case "SPII":
      return "blocked";
    case "MALFORMED_FUNCTION_CALL":
    case "OTHER":
      return "other";
    default:
      return "unknown";
  }
}
function fromGeminiThought(part) {
  return {
    reasoning: part.text || "",
    metadata: { thoughtSignature: part.thoughtSignature }
  };
}
function fromGeminiInlineData(part) {
  if (!part.inlineData || !part.inlineData.hasOwnProperty("mimeType") || !part.inlineData.hasOwnProperty("data")) {
    throw new Error("Invalid GeminiPart: missing required properties");
  }
  const { mimeType, data } = part.inlineData;
  const dataUrl = `data:${mimeType};base64,${data}`;
  return {
    media: {
      url: dataUrl,
      contentType: mimeType
    }
  };
}
function fromGeminiFileData(part) {
  if (!part.fileData || !part.fileData.hasOwnProperty("mimeType") || !part.fileData.hasOwnProperty("fileUri")) {
    throw new Error(
      "Invalid Gemini File Data Part: missing required properties"
    );
  }
  return {
    media: {
      url: part.fileData?.fileUri,
      contentType: part.fileData?.mimeType
    }
  };
}
function fromGeminiFunctionCall(part, ref) {
  if (!part.functionCall) {
    throw Error(
      "Invalid Gemini Function Call Part: missing function call data"
    );
  }
  return {
    toolRequest: {
      name: part.functionCall.name,
      input: part.functionCall.args,
      ref
    }
  };
}
function fromGeminiFunctionResponse(part, ref) {
  if (!part.functionResponse) {
    throw new Error(
      "Invalid Gemini Function Call Part: missing function call data"
    );
  }
  return {
    toolResponse: {
      name: part.functionResponse.name.replace(/__/g, "/"),
      // restore slashes
      output: part.functionResponse.response,
      ref
    }
  };
}
function fromExecutableCode(part) {
  if (!part.executableCode) {
    throw new Error("Invalid GeminiPart: missing executableCode");
  }
  return {
    custom: {
      executableCode: {
        language: part.executableCode.language,
        code: part.executableCode.code
      }
    }
  };
}
function fromCodeExecutionResult(part) {
  if (!part.codeExecutionResult) {
    throw new Error("Invalid GeminiPart: missing codeExecutionResult");
  }
  return {
    custom: {
      codeExecutionResult: {
        outcome: part.codeExecutionResult.outcome,
        output: part.codeExecutionResult.output
      }
    }
  };
}
function fromGeminiPart(part, ref) {
  if (part.thought) return fromGeminiThought(part);
  if (typeof part.text === "string") return { text: part.text };
  if (part.inlineData) return fromGeminiInlineData(part);
  if (part.fileData) return fromGeminiFileData(part);
  if (part.functionCall) return fromGeminiFunctionCall(part, ref);
  if (part.functionResponse) return fromGeminiFunctionResponse(part, ref);
  if (part.executableCode) return fromExecutableCode(part);
  if (part.codeExecutionResult) return fromCodeExecutionResult(part);
  throw new Error("Unsupported GeminiPart type " + JSON.stringify(part));
}
function fromGeminiCandidate(candidate) {
  const parts = candidate.content?.parts || [];
  const genkitCandidate = {
    index: candidate.index || 0,
    message: {
      role: "model",
      content: parts.filter((p) => Object.keys(p).length > 0).map((part, index) => fromGeminiPart(part, index.toString()))
    },
    finishReason: fromGeminiFinishReason(candidate.finishReason),
    finishMessage: candidate.finishMessage,
    custom: {
      safetyRatings: candidate.safetyRatings,
      citationMetadata: candidate.citationMetadata
    }
  };
  return genkitCandidate;
}
// Annotate the CommonJS export names for ESM import in node:
0 && (module.exports = {
  fromGeminiCandidate,
  toGeminiFunctionModeEnum,
  toGeminiMessage,
  toGeminiSystemInstruction,
  toGeminiTool
});
//# sourceMappingURL=converters.js.map