import { ModelReference, z, EmbedderReference, MediaPart, JSONSchema } from 'genkit';
import { GenerateRequest } from 'genkit/model';
import { GenerateContentStreamResult } from './types.js';

/**
 * Copyright 2025 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Safely extracts the error message from the error.
 * @param e The error
 * @returns The error message
 */
declare function extractErrMsg(e: unknown): string;
/**
 * Gets the un-prefixed model name from a modelReference
 */
declare function extractVersion(model: ModelReference<z.ZodTypeAny> | EmbedderReference<z.ZodTypeAny>): string;
/**
 * Gets the model name without certain prefixes..
 * e.g. for "models/googleai/gemini-2.5-pro" it returns just 'gemini-2.5-pro'
 * @param name A string containing the model string with possible prefixes
 * @returns the model string stripped of certain prefixes
 */
declare function modelName(name?: string): string | undefined;
/**
 * Gets the suffix of a model string.
 * Throws if the string is empty.
 * @param name A string containing the model string
 * @returns the model string stripped of prefixes and guaranteed not empty.
 */
declare function checkModelName(name?: string): string;
declare function extractText(request: GenerateRequest): string;
declare function extractMimeType(url?: string): string;
declare function checkSupportedMimeType(media: MediaPart['media'], supportedTypes: string[]): void;
/**
 *
 * @param url The url to show (e.g. in an error message)
 * @returns The appropriately  sized url
 */
declare function displayUrl(url: string): string;
/**
 *
 * @param request A generate request to extract from
 * @param metadataType The media must have metadata matching this type if isDefault is false
 * @param isDefault 'true' allows missing metadata type to match as well.
 * @returns
 */
declare function extractMedia(request: GenerateRequest, params: {
    metadataType?: string;
    isDefault?: boolean;
}): MediaPart['media'] | undefined;
/**
 * Cleans a JSON schema by removing specific keys and standardizing types.
 *
 * @param {JSONSchema} schema The JSON schema to clean.
 * @returns {JSONSchema} The cleaned JSON schema.
 */
declare function cleanSchema(schema: JSONSchema): JSONSchema;
/**
 * Processes the streaming body of a Response object. It decodes the stream as
 * UTF-8 text, parses JSON objects from specially formatted lines (e.g., "data: {}"),
 * and returns both an async generator for individual responses and a promise
 * that resolves to the aggregated final response.
 *
 * @param response The Response object with a streaming body.
 * @returns An object containing:
 *  - stream: An AsyncGenerator yielding each GenerateContentResponse.
 *  - response: A Promise resolving to the aggregated GenerateContentResponse.
 */
declare function processStream(response: Response): GenerateContentStreamResult;
declare function getGenkitClientHeader(): string;

export { checkModelName, checkSupportedMimeType, cleanSchema, displayUrl, extractErrMsg, extractMedia, extractMimeType, extractText, extractVersion, getGenkitClientHeader, modelName, processStream };
