import {
  embedderActionMetadata,
  z
} from "genkit";
import { embedderRef } from "genkit/embedder";
import { embedContent } from "./client.js";
import {
  TaskTypeSchema
} from "./types.js";
import {
  calculateApiKey,
  checkApiKey,
  checkModelName,
  extractVersion
} from "./utils.js";
const EmbeddingConfigSchema = z.object({
  /** Override the API key provided at plugin initialization. */
  apiKey: z.string().optional(),
  /**
   * The `task_type` parameter is defined as the intended downstream application to help the model
   * produce better quality embeddings.
   **/
  taskType: TaskTypeSchema.optional(),
  title: z.string().optional(),
  version: z.string().optional(),
  /**
   * The `outputDimensionality` parameter allows you to specify the dimensionality of the embedding output.
   * By default, the model generates embeddings with 768 dimensions. Models such as
   * `text-embedding-004`, `text-embedding-005`, and `text-multilingual-embedding-002`
   * allow the output dimensionality to be adjusted between 1 and 768.
   * By selecting a smaller output dimensionality, users can save memory and storage space, leading to more efficient computations.
   **/
  outputDimensionality: z.number().min(1).optional()
}).passthrough();
function commonRef(name, info, configSchema = EmbeddingConfigSchema) {
  return embedderRef({
    name: `googleai/${name}`,
    configSchema,
    info: info ?? {
      dimensions: 768,
      supports: {
        input: ["text"]
      }
    }
  });
}
const GENERIC_MODEL = commonRef("embedder");
const KNOWN_MODELS = {
  "text-embedding-004": commonRef("text-embedding-004"),
  "gemini-embedding-001": commonRef("gemini-embedding-001")
};
function model(version, config = {}) {
  const name = checkModelName(version);
  return embedderRef({
    name: `googleai/${name}`,
    config,
    configSchema: GENERIC_MODEL.configSchema,
    info: {
      ...GENERIC_MODEL.info
    }
  });
}
function listActions(models) {
  return models.filter((m) => m.supportedGenerationMethods.includes("embedContent")).filter((m) => !m.description || !m.description.includes("deprecated")).map((m) => {
    const ref = model(m.name);
    return embedderActionMetadata({
      name: ref.name,
      info: ref.info,
      configSchema: ref.configSchema
    });
  });
}
function defineKnownModels(ai, options) {
  for (const name of Object.keys(KNOWN_MODELS)) {
    defineEmbedder(ai, name, options);
  }
}
function defineEmbedder(ai, name, pluginOptions) {
  checkApiKey(pluginOptions?.apiKey);
  const ref = model(name);
  return ai.defineEmbedder(
    {
      name: ref.name,
      configSchema: ref.configSchema,
      info: ref.info
    },
    async (input, reqOptions) => {
      const embedApiKey = calculateApiKey(
        pluginOptions?.apiKey,
        reqOptions?.apiKey
      );
      const embedVersion = reqOptions?.version || extractVersion(ref);
      const embeddings = await Promise.all(
        input.map(async (doc) => {
          const response = await embedContent(embedApiKey, embedVersion, {
            taskType: reqOptions?.taskType,
            title: reqOptions?.title,
            content: {
              role: "",
              parts: [{ text: doc.text }]
            },
            outputDimensionality: reqOptions?.outputDimensionality
          });
          const values = response.embedding.values;
          return { embedding: values };
        })
      );
      return { embeddings };
    }
  );
}
const TEST_ONLY = { KNOWN_MODELS };
export {
  EmbeddingConfigSchema,
  TEST_ONLY,
  defineEmbedder,
  defineKnownModels,
  listActions,
  model
};
//# sourceMappingURL=embedder.mjs.map