import { logger } from "genkit/logging";
import { genkitPlugin } from "genkit/plugin";
import { extractErrMsg } from "../common/utils.js";
import { listModels } from "./client.js";
import { calculateApiKey } from "./utils.js";
import * as embedder from "./embedder.js";
import * as gemini from "./gemini.js";
import * as imagen from "./imagen.js";
import * as veo from "./veo.js";
async function initializer(ai, options) {
  imagen.defineKnownModels(ai, options);
  gemini.defineKnownModels(ai, options);
  embedder.defineKnownModels(ai, options);
  veo.defineKnownModels(ai, options);
}
async function resolver(ai, actionType, actionName, options) {
  switch (actionType) {
    case "model":
      if (veo.isVeoModelName(actionName)) {
      } else if (imagen.isImagenModelName(actionName)) {
        imagen.defineModel(ai, actionName, options);
      } else {
        gemini.defineModel(ai, actionName, options);
      }
      break;
    case "background-model":
      if (veo.isVeoModelName(actionName)) {
        veo.defineModel(ai, actionName, options);
      }
      break;
    case "embedder":
      embedder.defineEmbedder(ai, actionName, options);
      break;
    default:
  }
}
async function listActions(options) {
  try {
    const apiKey = calculateApiKey(options?.apiKey, void 0);
    const allModels = await listModels(apiKey, {
      baseUrl: options?.baseUrl,
      apiVersion: options?.apiVersion
    });
    return [
      ...gemini.listActions(allModels),
      ...imagen.listActions(allModels),
      ...veo.listActions(allModels),
      ...embedder.listActions(allModels)
    ];
  } catch (e) {
    logger.error(extractErrMsg(e));
    return [];
  }
}
function googleAIPlugin(options) {
  let listActionsCache;
  return genkitPlugin(
    "googleai",
    async (ai) => await initializer(ai, options),
    async (ai, actionType, actionName) => await resolver(ai, actionType, actionName, options || {}),
    async () => {
      if (listActionsCache) return listActionsCache;
      listActionsCache = await listActions(options);
      return listActionsCache;
    }
  );
}
const googleAI = googleAIPlugin;
googleAI.model = (name, config) => {
  if (veo.isVeoModelName(name)) {
    return veo.model(name, config);
  }
  if (imagen.isImagenModelName(name)) {
    return imagen.model(name, config);
  }
  return gemini.model(name, config);
};
googleAI.embedder = (name, config) => {
  return embedder.model(name, config);
};
var googleai_default = googleAI;
export {
  googleai_default as default,
  googleAI,
  googleAIPlugin
};
//# sourceMappingURL=index.mjs.map