import {
  modelActionMetadata,
  z
} from "genkit";
import {
  modelRef
} from "genkit/model";
import { veoCheckOperation, veoPredict } from "./client.js";
import {
  calculateApiKey,
  checkModelName,
  extractText,
  extractVeoImage,
  extractVersion,
  modelName
} from "./utils.js";
const VeoConfigSchema = z.object({
  // NOTE: Documentation notes numberOfVideos parameter to pick the number of
  // output videos, but this setting does not seem to work
  negativePrompt: z.string().optional(),
  aspectRatio: z.enum(["9:16", "16:9"]).describe("Desired aspect ratio of the output video.").optional(),
  personGeneration: z.enum(["dont_allow", "allow_adult", "allow_all"]).describe(
    "Control if/how images of people will be generated by the model."
  ).optional(),
  durationSeconds: z.number().step(1).min(5).max(8).describe("Length of each output video in seconds, between 5 and 8.").optional(),
  enhancePrompt: z.boolean().describe("Enable or disable the prompt rewriter. Enabled by default.").optional()
}).passthrough();
function commonRef(name, info, configSchema = VeoConfigSchema) {
  return modelRef({
    name: `googleai/${name}`,
    configSchema,
    info: info ?? {
      supports: {
        media: true,
        multiturn: false,
        tools: false,
        systemRole: false,
        output: ["media"],
        longRunning: true
      }
    }
    // TODO(ifielker): Remove this cast if we fix longRunning
  });
}
const GENERIC_MODEL = commonRef("veo");
const KNOWN_MODELS = {
  "veo-3.0-generate-preview": commonRef("veo-3.0-generate-preview"),
  "veo-3.0-fast-generate-preview": commonRef("veo-3.0-fast-generate-preview"),
  "veo-2.0-generate-001": commonRef("veo-2.0-generate-001")
};
function isVeoModelName(value) {
  return !!value?.startsWith("veo-");
}
function model(version, config = {}) {
  const name = checkModelName(version);
  return modelRef({
    name: `googleai/${name}`,
    config,
    configSchema: VeoConfigSchema,
    info: { ...GENERIC_MODEL.info }
  });
}
function listActions(models) {
  return models.filter(
    (m) => m.supportedGenerationMethods.includes("predictLongRunning") && isVeoModelName(modelName(m.name))
  ).filter((m) => !m.description || !m.description.includes("deprecated")).map((m) => {
    const ref = model(m.name);
    return modelActionMetadata({
      name: ref.name,
      info: ref.info,
      configSchema: ref.configSchema
    });
  });
}
function defineKnownModels(ai, options) {
  for (const name of Object.keys(KNOWN_MODELS)) {
    defineModel(ai, name, options);
  }
}
function defineModel(ai, name, pluginOptions) {
  const ref = model(name);
  const clientOptions = {
    apiVersion: pluginOptions?.apiVersion,
    baseUrl: pluginOptions?.baseUrl
  };
  return ai.defineBackgroundModel({
    name: ref.name,
    ...ref.info,
    configSchema: ref.configSchema,
    async start(request) {
      const apiKey = calculateApiKey(pluginOptions?.apiKey, void 0);
      const veoPredictRequest = {
        instances: [
          {
            prompt: extractText(request),
            image: extractVeoImage(request)
          }
        ],
        parameters: toVeoParameters(request)
      };
      const response = await veoPredict(
        apiKey,
        extractVersion(ref),
        veoPredictRequest,
        clientOptions
      );
      return fromVeoOperation(response);
    },
    async check(operation) {
      const apiKey = calculateApiKey(pluginOptions?.apiKey, void 0);
      const response = await veoCheckOperation(
        apiKey,
        operation.id,
        clientOptions
      );
      return fromVeoOperation(response);
    }
  });
}
function toVeoParameters(request) {
  const out = {
    ...request?.config
  };
  for (const k in out) {
    if (out[k] === null) delete out[k];
  }
  delete out.apiKey;
  delete out.version;
  return out;
}
function fromVeoOperation(apiOp) {
  const res = { id: apiOp.name };
  if (apiOp.done !== void 0) {
    res.done = apiOp.done;
  }
  if (apiOp.error) {
    res.error = { message: apiOp.error.message };
  }
  if (apiOp.response && apiOp.response.generateVideoResponse && apiOp.response.generateVideoResponse.generatedSamples) {
    res.output = {
      finishReason: "stop",
      raw: apiOp.response,
      message: {
        role: "model",
        content: apiOp.response.generateVideoResponse.generatedSamples.map(
          (s) => {
            return {
              media: {
                url: s.video.uri
              }
            };
          }
        )
      }
    };
  }
  return res;
}
const TEST_ONLY = {
  toVeoParameters,
  fromVeoOperation,
  GENERIC_MODEL,
  KNOWN_MODELS
};
export {
  TEST_ONLY,
  VeoConfigSchema,
  defineKnownModels,
  defineModel,
  isVeoModelName,
  listActions,
  model
};
//# sourceMappingURL=veo.mjs.map