import { z } from "genkit";
import {
  embedderRef
} from "genkit/embedder";
import { embedContent } from "./client.js";
import {
  TaskTypeSchema,
  isMultimodalEmbeddingPrediction,
  isObject
} from "./types.js";
import { checkModelName, extractVersion } from "./utils.js";
const EmbeddingConfigSchema = z.object({
  /**
   * The `task_type` parameter is defined as the intended downstream application
   * to help the model produce better quality embeddings.
   **/
  taskType: TaskTypeSchema.optional(),
  title: z.string().optional(),
  location: z.string().optional(),
  version: z.string().optional(),
  /**
   * The `outputDimensionality` parameter allows you to specify the dimensionality of the embedding output.
   * By default, the model generates embeddings with 768 dimensions.
   * By selecting a smaller output dimensionality, users can save memory and storage space, leading to more efficient computations.
   **/
  outputDimensionality: z.number().min(1).optional(),
  /**
   * For newly released embedders this parameter provides a hint for the proper
   * way to call the embedder. (Multimodal embedders have a different request
   * structure than non-multimodal embedders).
   * For well-known embedders, this value will be ignored since we will already
   * know if it's multimodal or not.
   */
  multimodal: z.boolean().optional()
}).passthrough();
function commonRef(name, info, configSchema = EmbeddingConfigSchema) {
  return embedderRef({
    name: `vertexai/${name}`,
    configSchema,
    info: info ?? {
      dimensions: 768,
      supports: {
        input: ["text"]
      }
    }
  });
}
const GENERIC_TEXT_MODEL = commonRef("text", {
  dimensions: 3072,
  supports: { input: ["text"] }
});
const GENERIC_MULTIMODAL_MODEL = commonRef("multimodal", {
  dimensions: 768,
  supports: { input: ["text", "image", "video"] }
});
const KNOWN_MODELS = {
  "text-embedding-005": commonRef("text-embedding-005"),
  "text-multilingual-embedding-002": commonRef(
    "text-multilingual-embedding-002"
  ),
  "multimodalembedding@001": commonRef("multimodalembedding@001", {
    dimensions: 768,
    supports: { input: ["text", "image", "video"] }
  }),
  "gemini-embedding-001": commonRef("gemini-embedding-001", {
    dimensions: 3072,
    supports: { input: ["text"] }
  })
};
function model(version, config = {}) {
  const name = checkModelName(version);
  if (KNOWN_MODELS[name]) {
    return embedderRef({
      name: `vertexai/${name}`,
      configSchema: EmbeddingConfigSchema,
      config,
      info: {
        ...KNOWN_MODELS[name].info
      }
    });
  }
  if (config.multimodal) {
    return embedderRef({
      name: `vertexai/${name}`,
      configSchema: EmbeddingConfigSchema,
      config,
      info: {
        ...GENERIC_MULTIMODAL_MODEL.info
      }
    });
  }
  return embedderRef({
    name: `vertexai/${name}`,
    configSchema: EmbeddingConfigSchema,
    config,
    info: {
      ...GENERIC_TEXT_MODEL.info
    }
  });
}
function defineKnownModels(ai, clientOptions, pluginOptions) {
  for (const name of Object.keys(KNOWN_MODELS)) {
    defineEmbedder(ai, name, clientOptions, pluginOptions);
  }
}
function defineEmbedder(ai, name, clientOptions, pluginOptions) {
  const ref = model(name);
  return ai.defineEmbedder(
    {
      name: ref.name,
      configSchema: ref.configSchema,
      info: ref.info
    },
    async (input, options) => {
      const embedContentRequest = {
        instances: input.map(
          (doc) => toEmbeddingInstance(ref, doc, options)
        ),
        parameters: { outputDimensionality: options?.outputDimensionality }
      };
      const response = await embedContent(
        extractVersion(ref),
        embedContentRequest,
        clientOptions
      );
      return {
        embeddings: response.predictions.map(toEmbeddingResult).reduce((accumulator, value) => {
          return accumulator.concat(value);
        }, [])
      };
    }
  );
}
function toEmbeddingInstance(embedder, doc, options) {
  let instance;
  if (isMultiModalEmbedder(embedder) || embedder.config?.multimodal || options?.multimodal) {
    instance = {};
    if (doc.text) {
      instance.text = doc.text;
    }
    for (var media of doc.media) {
      if (isObject(media) && typeof media.url === "string" && typeof media.contentType === "string") {
        if (media.contentType?.startsWith("image/")) {
          if (media.url.startsWith("http") || media.url.startsWith("gs://")) {
            instance.image = {
              gcsUri: media.url,
              mimeType: media.contentType
            };
          } else {
            instance.image = {
              bytesBase64Encoded: media.url,
              mimeType: media.contentType
            };
          }
        } else if (media.contentType.startsWith("video/")) {
          if (media.url.startsWith("http") || media.url.startsWith("gs://")) {
            instance.video = {
              gcsUri: media.url
            };
          } else {
            instance.video = {
              bytesBase64Encoded: media.url
            };
          }
          if (instance.video && doc.metadata && doc.metadata.videoSegmentConfig) {
            instance.video.videoSegmentConfig = doc.metadata.videoSegmentConfig;
          }
        } else {
          throw new Error(`Unsupported contentType: '${media.contentType}`);
        }
      } else {
        throw new Error("Invalid media specified.");
      }
    }
  } else {
    instance = {
      content: doc.text,
      task_type: options?.taskType,
      title: options?.title
    };
  }
  return instance;
}
function toEmbeddingResult(prediction) {
  if (isMultimodalEmbeddingPrediction(prediction)) {
    const eArray = [];
    if (prediction.imageEmbedding?.length) {
      const imageResult = {
        embedding: prediction.imageEmbedding,
        metadata: { embedType: "imageEmbedding" }
      };
      eArray.push(imageResult);
    }
    if (prediction.textEmbedding?.length) {
      const textResult = {
        embedding: prediction.textEmbedding,
        metadata: { embedType: "textEmbedding" }
      };
      eArray.push(textResult);
    }
    if (prediction.videoEmbeddings?.length) {
      for (const ve of prediction.videoEmbeddings) {
        if (ve.embedding?.length) {
          const { embedding, ...metadata } = ve;
          metadata.embedType = "videoEmbedding";
          const videoResult = {
            embedding,
            metadata
          };
          eArray.push(videoResult);
        }
      }
    }
    return eArray;
  } else {
    return [
      {
        embedding: prediction.embeddings.values
      }
    ];
  }
}
function isMultiModalEmbedder(embedder) {
  if (embedder.config?.multimodal) {
    return true;
  }
  const input = embedder.info?.supports?.input || "";
  return input.includes("text") && input.includes("image") || false;
}
const TEST_ONLY = { KNOWN_MODELS };
export {
  EmbeddingConfigSchema,
  KNOWN_MODELS,
  TEST_ONLY,
  defineEmbedder,
  defineKnownModels,
  model
};
//# sourceMappingURL=embedder.mjs.map