"use strict";
var __defProp = Object.defineProperty;
var __getOwnPropDesc = Object.getOwnPropertyDescriptor;
var __getOwnPropNames = Object.getOwnPropertyNames;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};
var __copyProps = (to, from, except, desc) => {
  if (from && typeof from === "object" || typeof from === "function") {
    for (let key of __getOwnPropNames(from))
      if (!__hasOwnProp.call(to, key) && key !== except)
        __defProp(to, key, { get: () => from[key], enumerable: !(desc = __getOwnPropDesc(from, key)) || desc.enumerable });
  }
  return to;
};
var __toCommonJS = (mod) => __copyProps(__defProp({}, "__esModule", { value: true }), mod);
var gemini_exports = {};
__export(gemini_exports, {
  GENERIC_MODEL: () => GENERIC_MODEL,
  GeminiConfigSchema: () => GeminiConfigSchema,
  KNOWN_MODELS: () => KNOWN_MODELS,
  SafetySettingsSchema: () => SafetySettingsSchema,
  TEST_ONLY: () => TEST_ONLY,
  defineKnownModels: () => defineKnownModels,
  defineModel: () => defineModel,
  isGeminiModelName: () => isGeminiModelName,
  listActions: () => listActions,
  model: () => model
});
module.exports = __toCommonJS(gemini_exports);
var import_genkit = require("genkit");
var import_model = require("genkit/model");
var import_middleware = require("genkit/model/middleware");
var import_tracing = require("genkit/tracing");
var import_converters = require("../common/converters.js");
var import_client = require("./client.js");
var import_converters2 = require("./converters.js");
var import_utils = require("./utils.js");
const SafetySettingsSchema = import_genkit.z.object({
  category: import_genkit.z.enum([
    /** The harm category is unspecified. */
    "HARM_CATEGORY_UNSPECIFIED",
    /** The harm category is hate speech. */
    "HARM_CATEGORY_HATE_SPEECH",
    /** The harm category is dangerous content. */
    "HARM_CATEGORY_DANGEROUS_CONTENT",
    /** The harm category is harassment. */
    "HARM_CATEGORY_HARASSMENT",
    /** The harm category is sexually explicit content. */
    "HARM_CATEGORY_SEXUALLY_EXPLICIT"
  ]),
  threshold: import_genkit.z.enum([
    "BLOCK_LOW_AND_ABOVE",
    "BLOCK_MEDIUM_AND_ABOVE",
    "BLOCK_ONLY_HIGH",
    "BLOCK_NONE"
  ])
});
const VertexRetrievalSchema = import_genkit.z.object({
  datastore: import_genkit.z.object({
    projectId: import_genkit.z.string().describe("Google Cloud Project ID.").optional(),
    location: import_genkit.z.string().describe("Google Cloud region e.g. us-central1.").optional(),
    dataStoreId: import_genkit.z.string().describe(
      'The data store id, when project id and location are provided as separate options. Alternatively, the full path to the data store should be provided in the form: "projects/{project}/locations/{location}/collections/default_collection/dataStores/{data_store}".'
    )
  }).describe("Vertex AI Search data store details"),
  disableAttribution: import_genkit.z.boolean().describe(
    "Disable using the search data in detecting grounding attribution. This does not affect how the result is given to the model for generation."
  ).optional()
});
const GoogleSearchRetrievalSchema = import_genkit.z.object({
  disableAttribution: import_genkit.z.boolean().describe(
    "Disable using the search data in detecting grounding attribution. This does not affect how the result is given to the model for generation."
  ).optional()
});
const GeminiConfigSchema = import_model.GenerationCommonConfigSchema.extend({
  apiKey: import_genkit.z.string().describe("Overrides the plugin-configured API key, if specified.").optional(),
  labels: import_genkit.z.record(import_genkit.z.string()).optional().describe("Key-value labels to attach to the request for cost tracking."),
  temperature: import_genkit.z.number().min(0).max(2).describe(
    import_model.GenerationCommonConfigDescriptions.temperature + " The default value is 1.0."
  ).optional(),
  topP: import_genkit.z.number().min(0).max(1).describe(
    import_model.GenerationCommonConfigDescriptions.topP + " The default value is 0.95."
  ).optional(),
  location: import_genkit.z.string().describe("Google Cloud region e.g. us-central1.").optional(),
  /**
   * Safety filter settings. See: https://cloud.google.com/vertex-ai/generative-ai/docs/multimodal/configure-safety-filters#configurable-filters
   *
   * E.g.
   *
   * ```js
   * config: {
   *   safetySettings: [
   *     {
   *       category: 'HARM_CATEGORY_HATE_SPEECH',
   *       threshold: 'BLOCK_LOW_AND_ABOVE',
   *     },
   *     {
   *       category: 'HARM_CATEGORY_DANGEROUS_CONTENT',
   *       threshold: 'BLOCK_MEDIUM_AND_ABOVE',
   *     },
   *     {
   *       category: 'HARM_CATEGORY_HARASSMENT',
   *       threshold: 'BLOCK_ONLY_HIGH',
   *     },
   *     {
   *       category: 'HARM_CATEGORY_SEXUALLY_EXPLICIT',
   *       threshold: 'BLOCK_NONE',
   *     },
   *   ],
   * }
   * ```
   */
  safetySettings: import_genkit.z.array(SafetySettingsSchema).describe(
    "Adjust how likely you are to see responses that could be harmful. Content is blocked based on the probability that it is harmful."
  ).optional(),
  /**
   * Vertex retrieval options.
   *
   * E.g.
   *
   * ```js
   *   config: {
   *     vertexRetrieval: {
   *       datastore: {
   *         projectId: 'your-cloud-project',
   *         location: 'us-central1',
   *         collection: 'your-collection',
   *       },
   *       disableAttribution: true,
   *     }
   *   }
   * ```
   */
  vertexRetrieval: VertexRetrievalSchema.describe(
    "Retrieve from Vertex AI Search data store for grounding generative responses."
  ).optional(),
  /**
   * Google Search retrieval options.
   *
   * ```js
   *   config: {
   *     googleSearchRetrieval: {
   *       disableAttribution: true,
   *     }
   *   }
   * ```
   */
  googleSearchRetrieval: GoogleSearchRetrievalSchema.describe(
    "Retrieve public web data for grounding, powered by Google Search."
  ).optional(),
  /**
   * Function calling options.
   *
   * E.g. forced tool call:
   *
   * ```js
   *   config: {
   *     functionCallingConfig: {
   *       mode: 'ANY',
   *     }
   *   }
   * ```
   */
  functionCallingConfig: import_genkit.z.object({
    mode: import_genkit.z.enum(["MODE_UNSPECIFIED", "AUTO", "ANY", "NONE"]).optional(),
    allowedFunctionNames: import_genkit.z.array(import_genkit.z.string()).optional()
  }).describe(
    "Controls how the model uses the provided tools (function declarations). With AUTO (Default) mode, the model decides whether to generate a natural language response or suggest a function call based on the prompt and context. With ANY, the model is constrained to always predict a function call and guarantee function schema adherence. With NONE, the model is prohibited from making function calls."
  ).optional(),
  thinkingConfig: import_genkit.z.object({
    includeThoughts: import_genkit.z.boolean().describe(
      "Indicates whether to include thoughts in the response.If true, thoughts are returned only if the model supports thought and thoughts are available."
    ).optional(),
    thinkingBudget: import_genkit.z.number().min(0).max(24576).describe(
      "Indicates the thinking budget in tokens. 0 is DISABLED. -1 is AUTOMATIC. The default values and allowed ranges are model dependent. The thinking budget parameter gives the model guidance on the number of thinking tokens it can use when generating a response. A greater number of tokens is typically associated with more detailed thinking, which is needed for solving more complex tasks. "
    ).optional()
  }).optional()
}).passthrough();
function commonRef(name, info, configSchema = GeminiConfigSchema) {
  return (0, import_model.modelRef)({
    name: `vertexai/${name}`,
    configSchema,
    info: info ?? {
      supports: {
        multiturn: true,
        media: true,
        tools: true,
        toolChoice: true,
        systemRole: true,
        constrained: "no-tools"
      }
    }
  });
}
const GENERIC_MODEL = commonRef("gemini");
const KNOWN_MODELS = {
  "gemini-2.5-flash-lite": commonRef("gemini-2.5-flash-lite"),
  "gemini-2.5-pro": commonRef("gemini-2.5-pro"),
  "gemini-2.5-flash": commonRef("gemini-2.5-flash"),
  "gemini-2.0-flash-001": commonRef("gemini-2.0-flash-001"),
  "gemini-2.0-flash": commonRef("gemini-2.0-flash"),
  "gemini-2.0-flash-lite": commonRef("gemini-2.0-flash-lite"),
  "gemini-2.0-flash-lite-001": commonRef("gemini-2.0-flash-lite-001")
};
function isGeminiModelName(value) {
  return !!value?.startsWith("gemini-") && !value.includes("embedding");
}
function model(version, options = {}) {
  const name = (0, import_utils.checkModelName)(version);
  return (0, import_model.modelRef)({
    name: `vertexai/${name}`,
    config: options,
    configSchema: GeminiConfigSchema,
    info: {
      ...GENERIC_MODEL.info
    }
  });
}
function listActions(models) {
  const KNOWN_DECOMISSIONED_MODELS = [
    "gemini-pro-vision",
    "gemini-pro",
    "gemini-ultra",
    "gemini-ultra-vision"
  ];
  return models.filter(
    (m) => isGeminiModelName((0, import_utils.modelName)(m.name)) && !KNOWN_DECOMISSIONED_MODELS.includes((0, import_utils.modelName)(m.name) || "")
  ).map((m) => {
    const ref = model(m.name);
    return (0, import_genkit.modelActionMetadata)({
      name: ref.name,
      info: ref.info,
      configSchema: ref.configSchema
    });
  });
}
function defineKnownModels(ai, clientOptions, pluginOptions) {
  for (const name of Object.keys(KNOWN_MODELS)) {
    defineModel(ai, name, clientOptions, pluginOptions);
  }
}
function defineModel(ai, name, clientOptions, pluginOptions) {
  const ref = model(name);
  const middlewares = [];
  if (ref.info?.supports?.media) {
    middlewares.push(
      (0, import_middleware.downloadRequestMedia)({
        maxBytes: 1024 * 1024 * 20,
        filter: (part) => {
          try {
            const url = new URL(part.media.url);
            if (
              // Gemini can handle these URLs
              ["www.youtube.com", "youtube.com", "youtu.be"].includes(
                url.hostname
              )
            )
              return false;
          } catch {
          }
          return true;
        }
      })
    );
  }
  return ai.defineModel(
    {
      apiVersion: "v2",
      name: ref.name,
      ...ref.info,
      configSchema: ref.configSchema,
      use: middlewares
    },
    async (request, { streamingRequested, sendChunk, abortSignal }) => {
      let clientOpt = { ...clientOptions, signal: abortSignal };
      const messages = structuredClone(request.messages);
      if (messages.length === 0) throw new Error("No messages provided.");
      let systemInstruction = void 0;
      const systemMessage = messages.find((m) => m.role === "system");
      if (systemMessage) {
        messages.splice(messages.indexOf(systemMessage), 1);
        systemInstruction = (0, import_converters.toGeminiSystemInstruction)(systemMessage);
      }
      const requestConfig = { ...request.config };
      const {
        apiKey: apiKeyFromConfig,
        functionCallingConfig,
        version: versionFromConfig,
        googleSearchRetrieval,
        tools: toolsFromConfig,
        vertexRetrieval,
        location,
        safetySettings,
        labels: labelsFromConfig,
        ...restOfConfig
      } = requestConfig;
      if (location && clientOptions.kind != "express" && clientOptions.location != location) {
        if (location == "global") {
          clientOpt = {
            kind: "global",
            location: "global",
            projectId: clientOptions.projectId,
            authClient: clientOptions.authClient,
            apiKey: clientOptions.apiKey,
            signal: abortSignal
          };
        } else {
          clientOpt = {
            kind: "regional",
            location,
            projectId: clientOptions.projectId,
            authClient: clientOptions.authClient,
            apiKey: clientOptions.apiKey,
            signal: abortSignal
          };
        }
      }
      if (clientOptions.kind == "express") {
        clientOpt.apiKey = (0, import_utils.calculateApiKey)(
          clientOptions.apiKey,
          apiKeyFromConfig
        );
      } else if (apiKeyFromConfig) {
        clientOpt.apiKey = apiKeyFromConfig;
      }
      const labels = (0, import_converters2.toGeminiLabels)(labelsFromConfig);
      const tools = [];
      if (request.tools?.length) {
        tools.push({
          functionDeclarations: request.tools.map(import_converters.toGeminiTool)
        });
      }
      let toolConfig;
      if (functionCallingConfig) {
        toolConfig = {
          functionCallingConfig: {
            allowedFunctionNames: functionCallingConfig.allowedFunctionNames,
            mode: (0, import_converters.toGeminiFunctionModeEnum)(functionCallingConfig.mode)
          }
        };
      } else if (request.toolChoice) {
        toolConfig = {
          functionCallingConfig: {
            mode: (0, import_converters.toGeminiFunctionModeEnum)(request.toolChoice)
          }
        };
      }
      const jsonMode = (request.output?.format === "json" || !!request.output?.schema) && tools.length === 0;
      if (toolsFromConfig) {
        tools.push(...toolsFromConfig);
      }
      if (googleSearchRetrieval) {
        if (ref.name.startsWith("vertexai/gemini-1.5")) {
          tools.push({
            googleSearchRetrieval
          });
        } else {
          tools.push({
            googleSearch: googleSearchRetrieval
          });
        }
      }
      if (vertexRetrieval) {
        const _projectId = vertexRetrieval.datastore.projectId || (clientOptions.kind != "express" ? clientOptions.projectId : void 0);
        const _location = vertexRetrieval.datastore.location || (clientOptions.kind == "regional" ? clientOptions.location : void 0);
        const _dataStoreId = vertexRetrieval.datastore.dataStoreId;
        if (!_projectId || !_location || !_dataStoreId) {
          throw new import_genkit.GenkitError({
            status: "INVALID_ARGUMENT",
            message: "projectId, location and datastoreId are required for vertexRetrieval and could not be determined from configuration"
          });
        }
        const datastore = `projects/${_projectId}/locations/${_location}/collections/default_collection/dataStores/${_dataStoreId}`;
        tools.push({
          retrieval: {
            vertexAiSearch: {
              datastore
            },
            disableAttribution: vertexRetrieval.disableAttribution
          }
        });
      }
      const generateContentRequest = {
        systemInstruction,
        generationConfig: {
          ...restOfConfig,
          candidateCount: request.candidates || void 0,
          responseMimeType: jsonMode ? "application/json" : void 0
        },
        tools,
        toolConfig,
        safetySettings: (0, import_converters2.toGeminiSafetySettings)(safetySettings),
        contents: messages.map((message) => (0, import_converters.toGeminiMessage)(message, ref)),
        labels
      };
      const modelVersion = versionFromConfig || (0, import_utils.extractVersion)(ref);
      if (jsonMode && request.output?.constrained) {
        generateContentRequest.generationConfig.responseSchema = (0, import_utils.cleanSchema)(
          request.output.schema
        );
      }
      const callGemini = async () => {
        let response;
        if (streamingRequested) {
          const result = await (0, import_client.generateContentStream)(
            modelVersion,
            generateContentRequest,
            clientOpt
          );
          for await (const item of result.stream) {
            item.candidates?.forEach(
              (candidate) => {
                const c = (0, import_converters.fromGeminiCandidate)(candidate);
                sendChunk({
                  index: c.index,
                  content: c.message.content
                });
              }
            );
          }
          response = await result.response;
        } else {
          response = await (0, import_client.generateContent)(
            modelVersion,
            generateContentRequest,
            clientOpt
          );
        }
        if (!response.candidates?.length) {
          throw new import_genkit.GenkitError({
            status: "FAILED_PRECONDITION",
            message: "No valid candidates returned."
          });
        }
        const candidateData = response.candidates.map(
          (c) => (0, import_converters.fromGeminiCandidate)(c)
        );
        return {
          candidates: candidateData,
          custom: response,
          usage: {
            ...(0, import_model.getBasicUsageStats)(request.messages, candidateData),
            inputTokens: response.usageMetadata?.promptTokenCount,
            outputTokens: response.usageMetadata?.candidatesTokenCount,
            thoughtsTokens: response.usageMetadata?.thoughtsTokenCount,
            totalTokens: response.usageMetadata?.totalTokenCount,
            cachedContentTokens: response.usageMetadata?.cachedContentTokenCount
          }
        };
      };
      const msg = (0, import_converters.toGeminiMessage)(messages[messages.length - 1], ref);
      return pluginOptions?.experimental_debugTraces ? await (0, import_tracing.runInNewSpan)(
        ai.registry,
        {
          metadata: {
            name: streamingRequested ? "sendMessageStream" : "sendMessage"
          }
        },
        async (metadata) => {
          metadata.input = {
            apiEndpoint: (0, import_client.getVertexAIUrl)({
              includeProjectAndLocation: false,
              resourcePath: "",
              clientOptions: clientOpt
            }),
            cache: {},
            model: modelVersion,
            generateContentOptions: generateContentRequest,
            parts: msg.parts,
            options: clientOpt
          };
          const response = await callGemini();
          metadata.output = response.custom;
          return response;
        }
      ) : await callGemini();
    }
  );
}
const TEST_ONLY = { KNOWN_MODELS };
// Annotate the CommonJS export names for ESM import in node:
0 && (module.exports = {
  GENERIC_MODEL,
  GeminiConfigSchema,
  KNOWN_MODELS,
  SafetySettingsSchema,
  TEST_ONLY,
  defineKnownModels,
  defineModel,
  isGeminiModelName,
  listActions,
  model
});
//# sourceMappingURL=gemini.js.map