import {
  modelActionMetadata,
  modelRef,
  z
} from "genkit";
import { lyriaPredict } from "./client.js";
import { fromLyriaResponse, toLyriaPredictRequest } from "./converters.js";
import { checkModelName, extractVersion } from "./utils.js";
const LyriaConfigSchema = z.object({
  negativePrompt: z.string().optional().describe(
    "Optional. A description of what to exclude from the generated audio."
  ),
  seed: z.number().optional().describe(
    "Optional. A seed for deterministic generation. If provided, the model will attempt to produce the same audio given the same prompt and other parameters. Cannot be used with sample_count in the same request."
  ),
  sampleCount: z.number().optional().describe(
    "Optional. The number of audio samples to generate. Default is 1 if not specified and seed is not used. Cannot be used with seed in the same request."
  )
}).passthrough();
function commonRef(name, info, configSchema = LyriaConfigSchema) {
  return modelRef({
    name: `vertexai/${name}`,
    configSchema,
    info: info ?? {
      supports: {
        media: true,
        multiturn: false,
        tools: false,
        systemRole: false,
        output: ["media"]
      }
    }
  });
}
const GENERIC_MODEL = commonRef("lyria");
const KNOWN_MODELS = {
  "lyria-002": commonRef("lyria-002")
};
function isLyriaModelName(value) {
  return !!value?.startsWith("lyria-");
}
function model(version, config = {}) {
  const name = checkModelName(version);
  return modelRef({
    name: `vertexai/${name}`,
    config,
    configSchema: LyriaConfigSchema,
    info: { ...GENERIC_MODEL.info }
  });
}
function listActions(models) {
  return models.filter((m) => isLyriaModelName(m.name)).map((m) => {
    const ref = model(m.name);
    return modelActionMetadata({
      name: ref.name,
      info: ref.info,
      configSchema: ref.configSchema
    });
  });
}
function defineKnownModels(ai, clientOptions, pluginOptions) {
  for (const name of Object.keys(KNOWN_MODELS)) {
    defineModel(ai, name, clientOptions, pluginOptions);
  }
}
function defineModel(ai, name, clientOptions, pluginOptions) {
  const ref = model(name);
  return ai.defineModel(
    {
      apiVersion: "v2",
      name: ref.name,
      ...ref.info,
      configSchema: ref.configSchema
    },
    async (request, { abortSignal }) => {
      const clientOpt = { ...clientOptions, signal: abortSignal };
      const lyriaPredictRequest = toLyriaPredictRequest(request);
      const response = await lyriaPredict(
        extractVersion(ref),
        lyriaPredictRequest,
        clientOpt
      );
      if (!response.predictions || response.predictions.length == 0) {
        throw new Error(
          "Model returned no predictions. Possibly due to content filters."
        );
      }
      return fromLyriaResponse(response, request);
    }
  );
}
const TEST_ONLY = {
  GENERIC_MODEL,
  KNOWN_MODELS
};
export {
  LyriaConfigSchema,
  TEST_ONLY,
  defineKnownModels,
  defineModel,
  isLyriaModelName,
  listActions,
  model
};
//# sourceMappingURL=lyria.mjs.map