import {
  modelActionMetadata,
  modelRef,
  z
} from "genkit";
import { veoCheckOperation, veoPredict } from "./client.js";
import {
  fromVeoOperation,
  toVeoModel,
  toVeoOperationRequest,
  toVeoPredictRequest
} from "./converters.js";
import { checkModelName, extractVersion } from "./utils.js";
const VeoConfigSchema = z.object({
  sampleCount: z.number().optional().describe("Number of output videos"),
  storageUri: z.string().optional().describe("The gcs bucket where to save the generated videos"),
  fps: z.number().optional().describe("Frames per second for video generation"),
  durationSeconds: z.number().optional().describe("Duration of the clip for video generation in seconds"),
  seed: z.number().optional().describe(
    "The RNG seed. If RNG seed is exactly same for each request with unchanged inputs, the prediction results will be consistent. Otherwise, a random RNG seed will be used each time to produce a different result. If the sample count is greater than 1, random seeds will be used for each sample."
  ),
  aspectRatio: z.enum(["9:16", "16:9"]).optional().describe("The aspect ratio for the generated video"),
  resolution: z.enum(["720p", "1080p"]).optional().describe("The resolution for the generated video"),
  personGeneration: z.enum(["dont_allow", "allow_adult", "allow_all"]).optional().describe(
    "Specifies the policy for generating persons in videos, including age restrictions"
  ),
  pubsubTopic: z.string().optional().describe("The pubsub topic to publish the video generation progress to"),
  negativePrompt: z.string().optional().describe(
    "In addition to the text context, negative prompts can be explicitly stated here to help generate the video"
  ),
  enhancePrompt: z.boolean().optional().describe(
    "If true, the prompt will be improved before it is used to generate videos. The RNG seed, if provided, will not result in consistent results if prompts are enhanced."
  ),
  generateAudio: z.boolean().optional().describe("If true, audio will be generated along with the video"),
  compressionQuality: z.enum(["optimized", "lossless"]).default("optimized").optional().describe("Compression quality of the generated video")
}).passthrough();
function commonRef(name, info, configSchema = VeoConfigSchema) {
  return modelRef({
    name: `vertexai/${name}`,
    configSchema,
    info: info ?? {
      supports: {
        media: true,
        multiturn: false,
        tools: false,
        systemRole: false,
        output: ["media"],
        longRunning: true
      }
    }
    // TODO(ifielker): Remove this cast if we fix longRunning
  });
}
const GENERIC_MODEL = commonRef("veo");
const KNOWN_MODELS = {
  "veo-2.0-generate-001": commonRef("veo-2.0-generate-001"),
  "veo-3.0-generate-001": commonRef("veo-3.0-generate-001"),
  "veo-3.0-fast-generate-001": commonRef("veo-3.0-fast-generate-001"),
  "veo-3.0-generate-preview": commonRef("veo-3.0-generate-preview"),
  "veo-3.0-fast-generate-preview": commonRef("veo-3.0-fast-generate-preview")
};
function isVeoModelName(value) {
  return !!value?.startsWith("veo-");
}
function model(version, config = {}) {
  const name = checkModelName(version);
  return modelRef({
    name: `vertexai/${name}`,
    config,
    configSchema: VeoConfigSchema,
    info: { ...GENERIC_MODEL.info }
  });
}
function listActions(models) {
  return models.filter((m) => isVeoModelName(m.name)).map((m) => {
    const ref = model(m.name);
    return modelActionMetadata({
      name: ref.name,
      info: ref.info,
      configSchema: ref.configSchema
    });
  });
}
function defineKnownModels(ai, clientOptions, pluginOptions) {
  for (const name of Object.keys(KNOWN_MODELS)) {
    defineModel(ai, name, clientOptions, pluginOptions);
  }
}
function defineModel(ai, name, clientOptions, pluginOptions) {
  const ref = model(name);
  return ai.defineBackgroundModel({
    name: ref.name,
    ...ref.info,
    configSchema: ref.configSchema,
    async start(request) {
      const veoPredictRequest = toVeoPredictRequest(request);
      const response = await veoPredict(
        extractVersion(ref),
        veoPredictRequest,
        clientOptions
      );
      return fromVeoOperation(response);
    },
    async check(operation) {
      const response = await veoCheckOperation(
        toVeoModel(operation),
        toVeoOperationRequest(operation),
        clientOptions
      );
      return fromVeoOperation(response);
    }
  });
}
const TEST_ONLY = {
  GENERIC_MODEL,
  KNOWN_MODELS
};
export {
  TEST_ONLY,
  VeoConfigSchema,
  defineKnownModels,
  defineModel,
  isVeoModelName,
  listActions,
  model
};
//# sourceMappingURL=veo.mjs.map