import axios from 'axios';
import chokidar from 'chokidar';
import EventEmitter from 'events';
import fs from 'fs/promises';
import path from 'path';
import { RunActionResponseSchema, } from '../types/action';
import * as apis from '../types/apis';
import { logger } from '../utils/logger';
import { checkServerHealth, findRuntimesDir, findServersDir, isValidDevToolsInfo, projectNameFromGenkitFilePath, removeToolsInfoFile, retriable, } from '../utils/utils';
import { GenkitToolsError, RuntimeEvent, } from './types';
const STREAM_DELIMITER = '\n';
const HEALTH_CHECK_INTERVAL = 5000;
export const GENKIT_REFLECTION_API_SPEC_VERSION = 1;
export class RuntimeManager {
    telemetryServerUrl;
    manageHealth;
    projectRoot;
    filenameToRuntimeMap = {};
    filenameToDevUiMap = {};
    idToFileMap = {};
    eventEmitter = new EventEmitter();
    constructor(telemetryServerUrl, manageHealth, projectRoot) {
        this.telemetryServerUrl = telemetryServerUrl;
        this.manageHealth = manageHealth;
        this.projectRoot = projectRoot;
    }
    static async create(options) {
        const manager = new RuntimeManager(options.telemetryServerUrl, options.manageHealth ?? true, options.projectRoot);
        await manager.setupRuntimesWatcher();
        await manager.setupDevUiWatcher();
        if (manager.manageHealth) {
            setInterval(async () => await manager.performHealthChecks(), HEALTH_CHECK_INTERVAL);
        }
        return manager;
    }
    listRuntimes() {
        return Object.values(this.filenameToRuntimeMap);
    }
    getRuntimeById(id) {
        const fileName = this.idToFileMap[id];
        return fileName ? this.filenameToRuntimeMap[fileName] : undefined;
    }
    getMostRecentRuntime() {
        const runtimes = Object.values(this.filenameToRuntimeMap);
        return runtimes.length === 0
            ? undefined
            : runtimes.reduce((a, b) => new Date(a.timestamp) > new Date(b.timestamp) ? a : b);
    }
    getMostRecentDevUI() {
        const toolsInfo = Object.values(this.filenameToDevUiMap);
        return toolsInfo.length === 0
            ? undefined
            : toolsInfo.reduce((a, b) => new Date(a.timestamp) > new Date(b.timestamp) ? a : b);
    }
    onRuntimeEvent(listener) {
        Object.values(RuntimeEvent).forEach((event) => this.eventEmitter.on(event, (rt) => listener(event, rt)));
    }
    async listActions(input) {
        const runtime = input?.runtimeId
            ? this.getRuntimeById(input.runtimeId)
            : this.getMostRecentRuntime();
        if (!runtime) {
            throw new Error(input?.runtimeId
                ? `No runtime found with ID ${input.runtimeId}.`
                : 'No runtimes found. Make sure your app is running using `genkit start -- ...`. See getting started documentation.');
        }
        const response = await axios
            .get(`${runtime.reflectionServerUrl}/api/actions`)
            .catch((err) => this.httpErrorHandler(err, 'Error listing actions.'));
        return response.data;
    }
    async runAction(input, streamingCallback) {
        const runtime = input.runtimeId
            ? this.getRuntimeById(input.runtimeId)
            : this.getMostRecentRuntime();
        if (!runtime) {
            throw new Error(input.runtimeId
                ? `No runtime found with ID ${input.runtimeId}.`
                : 'No runtimes found. Make sure your app is running using `genkit start -- ...`. See getting started documentation.');
        }
        if (streamingCallback) {
            const response = await axios
                .post(`${runtime.reflectionServerUrl}/api/runAction?stream=true`, input, {
                headers: {
                    'Content-Type': 'application/json',
                },
                responseType: 'stream',
            })
                .catch(this.httpErrorHandler);
            let genkitVersion;
            if (response.headers['x-genkit-version']) {
                genkitVersion = response.headers['x-genkit-version'];
            }
            const stream = response.data;
            let buffer = '';
            stream.on('data', (data) => {
                buffer += data;
                while (buffer.includes(STREAM_DELIMITER)) {
                    try {
                        streamingCallback(JSON.parse(buffer.substring(0, buffer.indexOf(STREAM_DELIMITER))));
                        buffer = buffer.substring(buffer.indexOf(STREAM_DELIMITER) + STREAM_DELIMITER.length);
                    }
                    catch (err) {
                        logger.error(`Bad stream: ${err}`);
                        break;
                    }
                }
            });
            let resolver;
            let rejecter;
            const promise = new Promise((resolve, reject) => {
                resolver = resolve;
                rejecter = reject;
            });
            stream.on('end', () => {
                const parsedBuffer = JSON.parse(buffer);
                if (parsedBuffer.error) {
                    const err = new GenkitToolsError(`Error running action key='${input.key}'.`);
                    err.data = {
                        ...parsedBuffer.error,
                        stack: (parsedBuffer.error?.details).stack,
                        data: {
                            genkitErrorMessage: parsedBuffer.error?.message,
                            genkitErrorDetails: parsedBuffer.error?.details,
                        },
                    };
                    rejecter(err);
                    return;
                }
                const actionResponse = RunActionResponseSchema.parse(parsedBuffer);
                if (genkitVersion) {
                    actionResponse.genkitVersion = genkitVersion;
                }
                resolver(actionResponse);
            });
            stream.on('error', (err) => {
                rejecter(err);
            });
            return promise;
        }
        else {
            const response = await axios
                .post(`${runtime.reflectionServerUrl}/api/runAction`, input, {
                headers: {
                    'Content-Type': 'application/json',
                },
            })
                .catch((err) => this.httpErrorHandler(err, `Error running action key='${input.key}'.`));
            const resp = RunActionResponseSchema.parse(response.data);
            if (response.headers['x-genkit-version']) {
                resp.genkitVersion = response.headers['x-genkit-version'];
            }
            return resp;
        }
    }
    async listTraces(input) {
        const { limit, continuationToken, filter } = input;
        let query = '';
        if (limit) {
            query += `limit=${limit}`;
        }
        if (continuationToken) {
            if (query !== '') {
                query += '&';
            }
            query += `continuationToken=${continuationToken}`;
        }
        if (filter) {
            if (query !== '') {
                query += '&';
            }
            query += `filter=${encodeURI(JSON.stringify(filter))}`;
        }
        const response = await axios
            .get(`${this.telemetryServerUrl}/api/traces?${query}`)
            .catch((err) => this.httpErrorHandler(err, `Error listing traces for query='${query}'.`));
        return apis.ListTracesResponseSchema.parse(response.data);
    }
    async getTrace(input) {
        const { traceId } = input;
        const response = await axios
            .get(`${this.telemetryServerUrl}/api/traces/${traceId}`)
            .catch((err) => this.httpErrorHandler(err, `Error getting trace for traceId='${traceId}'`));
        return response.data;
    }
    async notifyRuntime(runtime) {
        try {
            await axios.post(`${runtime.reflectionServerUrl}/api/notify`, {
                telemetryServerUrl: this.telemetryServerUrl,
                reflectionApiSpecVersion: GENKIT_REFLECTION_API_SPEC_VERSION,
            });
        }
        catch (error) {
            logger.error(`Failed to notify runtime ${runtime.id}: ${error}`);
        }
    }
    async setupRuntimesWatcher() {
        try {
            const runtimesDir = await findRuntimesDir(this.projectRoot);
            await fs.mkdir(runtimesDir, { recursive: true });
            const watcher = chokidar.watch(runtimesDir, {
                persistent: true,
                ignoreInitial: false,
            });
            watcher.on('add', (filePath) => this.handleNewRuntime(filePath));
            if (this.manageHealth) {
                watcher.on('unlink', (filePath) => this.handleRemovedRuntime(filePath));
            }
            for (const runtime of await fs.readdir(runtimesDir)) {
                await this.handleNewRuntime(path.resolve(runtimesDir, runtime));
            }
        }
        catch (error) {
            logger.error('Failed to set up runtimes watcher:', error);
        }
    }
    async setupDevUiWatcher() {
        try {
            const serversDir = await findServersDir(this.projectRoot);
            await fs.mkdir(serversDir, { recursive: true });
            const watcher = chokidar.watch(serversDir, {
                persistent: true,
                ignoreInitial: false,
            });
            watcher.on('add', (filePath) => this.handleNewDevUi(filePath));
            if (this.manageHealth) {
                watcher.on('unlink', (filePath) => this.handleRemovedDevUi(filePath));
            }
            for (const toolsInfo of await fs.readdir(serversDir)) {
                await this.handleNewDevUi(path.resolve(serversDir, toolsInfo));
            }
        }
        catch (error) {
            logger.error('Failed to set up tools server watcher:', error);
        }
    }
    async handleNewDevUi(filePath) {
        try {
            const { content, toolsInfo } = await retriable(async () => {
                const content = await fs.readFile(filePath, 'utf-8');
                const toolsInfo = JSON.parse(content);
                return { content, toolsInfo };
            }, { maxRetries: 10, delayMs: 500 });
            if (isValidDevToolsInfo(toolsInfo)) {
                const fileName = path.basename(filePath);
                if (await checkServerHealth(toolsInfo.url)) {
                    this.filenameToDevUiMap[fileName] = toolsInfo;
                }
                else {
                    logger.debug('Found an unhealthy tools config file', fileName);
                    await removeToolsInfoFile(fileName, this.projectRoot);
                }
            }
            else {
                logger.error(`Unexpected file in the servers directory: ${content}`);
            }
        }
        catch (error) {
            logger.error('Error reading tools config', error);
            return undefined;
        }
    }
    handleRemovedDevUi(filePath) {
        const fileName = path.basename(filePath);
        if (fileName in this.filenameToDevUiMap) {
            const toolsInfo = this.filenameToDevUiMap[fileName];
            delete this.filenameToDevUiMap[fileName];
            logger.debug(`Removed Dev UI with url ${toolsInfo.url}.`);
        }
    }
    async handleNewRuntime(filePath) {
        try {
            const { content, runtimeInfo } = await retriable(async () => {
                const content = await fs.readFile(filePath, 'utf-8');
                const runtimeInfo = JSON.parse(content);
                runtimeInfo.projectName = projectNameFromGenkitFilePath(filePath);
                return { content, runtimeInfo };
            }, { maxRetries: 10, delayMs: 500 });
            if (isValidRuntimeInfo(runtimeInfo)) {
                if (!runtimeInfo.name) {
                    runtimeInfo.name = runtimeInfo.id;
                }
                const fileName = path.basename(filePath);
                if (await checkServerHealth(runtimeInfo.reflectionServerUrl)) {
                    if (runtimeInfo.reflectionApiSpecVersion !=
                        GENKIT_REFLECTION_API_SPEC_VERSION) {
                        if (!runtimeInfo.reflectionApiSpecVersion ||
                            runtimeInfo.reflectionApiSpecVersion <
                                GENKIT_REFLECTION_API_SPEC_VERSION) {
                            logger.warn('Genkit CLI is newer than runtime library. Some feature may not be supported. ' +
                                'Consider upgrading your runtime library version (debug info: expected ' +
                                `${GENKIT_REFLECTION_API_SPEC_VERSION}, got ${runtimeInfo.reflectionApiSpecVersion}).`);
                        }
                        else {
                            logger.error('Genkit CLI version is outdated. Please update `genkit-cli` to the latest version.');
                            process.exit(1);
                        }
                    }
                    this.filenameToRuntimeMap[fileName] = runtimeInfo;
                    this.idToFileMap[runtimeInfo.id] = fileName;
                    this.eventEmitter.emit(RuntimeEvent.ADD, runtimeInfo);
                    await this.notifyRuntime(runtimeInfo);
                    logger.debug(`Added runtime with ID ${runtimeInfo.id} at URL: ${runtimeInfo.reflectionServerUrl}`);
                }
                else {
                    await this.removeRuntime(fileName);
                }
            }
            else {
                logger.error(`Unexpected file in the runtimes directory: ${content}`);
            }
        }
        catch (error) {
            logger.error(`Error processing file ${filePath}:`, error);
        }
    }
    handleRemovedRuntime(filePath) {
        const fileName = path.basename(filePath);
        if (fileName in this.filenameToRuntimeMap) {
            const runtime = this.filenameToRuntimeMap[fileName];
            delete this.filenameToRuntimeMap[fileName];
            delete this.idToFileMap[runtime.id];
            this.eventEmitter.emit(RuntimeEvent.REMOVE, runtime);
            logger.debug(`Removed runtime with id ${runtime.id}.`);
        }
    }
    httpErrorHandler(error, message) {
        const newError = new GenkitToolsError(message || 'Internal Error');
        if (error.response) {
            if ((error.response?.data).message) {
                newError.message = (error.response?.data).message;
            }
            newError.data = error.response.data;
            throw newError;
        }
        throw new GenkitToolsError(message || 'Internal Error', {
            cause: error.cause,
        });
    }
    async performHealthChecks() {
        const healthCheckPromises = Object.entries(this.filenameToRuntimeMap).map(async ([fileName, runtime]) => {
            if (!(await checkServerHealth(runtime.reflectionServerUrl))) {
                await this.removeRuntime(fileName);
            }
        });
        return Promise.all(healthCheckPromises);
    }
    async removeRuntime(fileName) {
        const runtime = this.filenameToRuntimeMap[fileName];
        if (runtime) {
            try {
                const runtimesDir = await findRuntimesDir(this.projectRoot);
                const runtimeFilePath = path.join(runtimesDir, fileName);
                await fs.unlink(runtimeFilePath);
            }
            catch (error) {
                logger.debug(`Failed to delete runtime file: ${error}`);
            }
            logger.debug(`Removed unhealthy runtime with ID ${runtime.id} from manager.`);
        }
    }
}
function isValidRuntimeInfo(data) {
    let timestamp = '';
    if (typeof data.timestamp === 'string') {
        timestamp = data.timestamp.replaceAll('_', ':');
    }
    return (typeof data === 'object' &&
        data !== null &&
        typeof data.id === 'string' &&
        typeof data.pid === 'number' &&
        typeof data.reflectionServerUrl === 'string' &&
        typeof data.timestamp === 'string' &&
        !isNaN(Date.parse(timestamp)) &&
        (data.name === undefined || typeof data.name === 'string'));
}
//# sourceMappingURL=manager.js.map