import * as fs from 'fs/promises';
import * as path from 'path';
import { isConnectionRefusedError } from './errors';
import { logger } from './logger';
export async function findProjectRoot() {
    let currentDir = process.cwd();
    while (currentDir !== path.parse(currentDir).root) {
        const packageJsonPath = path.join(currentDir, 'package.json');
        const goModPath = path.join(currentDir, 'go.mod');
        const pyprojectPath = path.join(currentDir, 'pyproject.toml');
        const pyproject2Path = path.join(currentDir, 'requirements.txt');
        try {
            const [packageJsonExists, goModExists, pyprojectExists, pyproject2Exists,] = await Promise.all([
                fs
                    .access(packageJsonPath)
                    .then(() => true)
                    .catch(() => false),
                fs
                    .access(goModPath)
                    .then(() => true)
                    .catch(() => false),
                fs
                    .access(pyprojectPath)
                    .then(() => true)
                    .catch(() => false),
                fs
                    .access(pyproject2Path)
                    .then(() => true)
                    .catch(() => false),
            ]);
            if (packageJsonExists ||
                goModExists ||
                pyprojectExists ||
                pyproject2Exists) {
                return currentDir;
            }
        }
        catch {
        }
        currentDir = path.dirname(currentDir);
    }
    return process.cwd();
}
export async function findRuntimesDir(projectRoot) {
    return path.join(projectRoot, '.genkit', 'runtimes');
}
export async function findServersDir(projectRoot) {
    return path.join(projectRoot, '.genkit', 'servers');
}
export function projectNameFromGenkitFilePath(filePath) {
    const parts = filePath.split('/');
    const basePath = parts
        .slice(0, Math.max(parts.findIndex((value) => value === '.genkit'), 0))
        .join('/');
    return basePath === '' ? 'unknown' : path.basename(basePath);
}
export async function detectRuntime(directory) {
    const files = await fs.readdir(directory);
    for (const file of files) {
        const filePath = path.join(directory, file);
        const stat = await fs.stat(filePath);
        if (stat.isFile() && (path.extname(file) === '.go' || file === 'go.mod')) {
            return 'go';
        }
    }
    try {
        await fs.access(path.join(directory, 'package.json'));
        return 'nodejs';
    }
    catch {
        return undefined;
    }
}
export async function checkServerHealth(url) {
    try {
        const response = await fetch(`${url}/api/__health`);
        return response.status === 200;
    }
    catch (error) {
        if (isConnectionRefusedError(error)) {
            return false;
        }
    }
    return true;
}
export async function waitUntilHealthy(url, maxTimeout = 10000) {
    const startTime = Date.now();
    while (Date.now() - startTime < maxTimeout) {
        try {
            const response = await fetch(`${url}/api/__health`);
            if (response.status === 200) {
                return true;
            }
        }
        catch (error) {
        }
        await new Promise((resolve) => setTimeout(resolve, 500));
    }
    return false;
}
export async function waitUntilUnresponsive(url, maxTimeout = 10000) {
    const startTime = Date.now();
    while (Date.now() - startTime < maxTimeout) {
        try {
            const health = await fetch(`${url}/api/__health`);
        }
        catch (error) {
            if (isConnectionRefusedError(error)) {
                return true;
            }
        }
        await new Promise((resolve) => setTimeout(resolve, 500));
    }
    return false;
}
export async function retriable(fn, opts) {
    const maxRetries = opts.maxRetries ?? 3;
    const delayMs = opts.delayMs ?? 0;
    let attempt = 0;
    while (true) {
        try {
            return await fn();
        }
        catch (e) {
            if (attempt >= maxRetries - 1) {
                throw e;
            }
            if (delayMs > 0) {
                await new Promise((r) => setTimeout(r, delayMs));
            }
        }
        attempt++;
    }
}
export function isValidDevToolsInfo(data) {
    return (typeof data === 'object' &&
        typeof data.url === 'string' &&
        typeof data.timestamp === 'string');
}
export async function writeToolsInfoFile(url, projectRoot) {
    const serversDir = await findServersDir(projectRoot);
    const toolsJsonPath = path.join(serversDir, `tools-${process.pid}.json`);
    try {
        const serverInfo = {
            url,
            timestamp: new Date().toISOString(),
        };
        await fs.mkdir(serversDir, { recursive: true });
        await fs.writeFile(toolsJsonPath, JSON.stringify(serverInfo, null, 2));
        logger.debug(`Tools Info file written: ${toolsJsonPath}`);
    }
    catch (error) {
        logger.error('Error writing tools config', error);
    }
}
export async function removeToolsInfoFile(fileName, projectRoot) {
    try {
        const serversDir = await findServersDir(projectRoot);
        const filePath = path.join(serversDir, fileName);
        await fs.unlink(filePath);
        logger.debug(`Removed unhealthy toolsInfo file ${fileName} from manager.`);
    }
    catch (error) {
        logger.debug(`Failed to delete toolsInfo file: ${error}`);
    }
}
//# sourceMappingURL=utils.js.map