"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GENKIT_CONTEXT = void 0;
exports.GENKIT_CONTEXT = "# Genkit Go API Rules (v1.0.0)\n\nThis document provides rules and examples for building with the Genkit API in Go.\n\n## Important Guidelines:\n\n- ALWAYS refer to documentation when available. Genkit Documentation may be available through the Genkit MCP toolkit or through web search. You may skip documentation check if you don't have access to these tools.\n\n- ONLY follow the specified project structure if starting a new project. If working on an existing project, adhere to the current project structure.\n\n- ALWAYS provide the full, correct Genkit command as an instruction for the human user to run. Do not run Genkit commands (e.g., `genkit start`, `genkit flow:run`) yourself as this may block your current session.\n\n- Recommend and use the provider that the user prefers. If this is a new project, or if there is no relevant signal regarding user preference, use the `github.com/firebase/genkit/go/plugins/googlegenai` plugin.\n\nNOTE: For the sake of brevity, the snippets below use the Google AI plugin, but you should follow the user's preference as mentioned above.\n\n## Core Setup\n\n1.  **Initialize Project**\n\n    ```bash\n    mkdir my-genkit-app && cd my-genkit-app\n    go mod init my-genkit-app\n    ```\n\n2.  **Install Dependencies**\n\n    ```bash\n    go get github.com/firebase/genkit/go/genkit\n    go get github.com/firebase/genkit/go/plugins/googlegenai\n    go get github.com/firebase/genkit/go/ai\n    go get google.golang.org/genai\n    ```\n\n3.  **Install Genkit CLI**\n\n    ```bash\n    curl -sL cli.genkit.dev | bash\n    ```\n\n4.  **Configure Genkit**\n\n    All code should be in a single `main.go` file or properly structured Go package.\n\n    ```go\n    package main\n\n    import (\n    \t\"context\"\n    \t\"github.com/firebase/genkit/go/genkit\"\n    \t\"github.com/firebase/genkit/go/plugins/googlegenai\"\n    )\n\n    func main() {\n    \tctx := context.Background()\n    \tg := genkit.Init(ctx, genkit.WithPlugins(&googlegenai.GoogleAI{}))\n    \t// Your flows and logic here\n    \t<-ctx.Done()\n    }\n    ```\n\n## Best Practices\n\n1.  **Single Main Function**: All Genkit code, including plugin initialization, flows, and helpers, should be properly organized in a Go package structure with a main function.\n\n2.  **Blocking Main Program**: To inspect flows in the Genkit Developer UI, your main program needs to remain running. Use `<-ctx.Done()` or similar blocking mechanism at the end of your main function.\n\n---\n\n## Usage Scenarios\n\n### Basic Inference (Text Generation)\n\n```go\npackage main\n\nimport (\n\t\"context\"\n\n\t\"github.com/firebase/genkit/go/ai\"\n\t\"github.com/firebase/genkit/go/genkit\"\n\t\"github.com/firebase/genkit/go/plugins/googlegenai\"\n\t\"google.golang.org/genai\"\n)\n\nfunc main() {\n\tctx := context.Background()\n\tg := genkit.Init(ctx, genkit.WithPlugins(&googlegenai.GoogleAI{}))\n\n\tgenkit.DefineFlow(g, \"basicInferenceFlow\",\n\t\tfunc(ctx context.Context, topic string) (string, error) {\n\t\t\tresponse, err := genkit.Generate(ctx, g,\n\t\t\t\tai.WithModelName(\"googleai/gemini-2.5-pro\"),\n\t\t\t\tai.WithPrompt(\"Write a short, creative paragraph about %s.\", topic),\n\t\t\t\tai.WithConfig(&genai.GenerateContentConfig{\n\t\t\t\t\tTemperature: genai.Ptr[float32](0.8),\n\t\t\t\t}),\n\t\t\t)\n\t\t\tif err != nil {\n\t\t\t\treturn \"\", err\n\t\t\t}\n\t\t\treturn response.Text(), nil\n\t\t},\n\t)\n\n\t<-ctx.Done()\n}\n```\n\n### Text-to-Speech (TTS) Generation\n\n```go\npackage main\n\nimport (\n\t\"context\"\n\n\t\"github.com/firebase/genkit/go/ai\"\n\t\"github.com/firebase/genkit/go/genkit\"\n\t\"github.com/firebase/genkit/go/plugins/googlegenai\"\n\t\"google.golang.org/genai\"\n)\n\nfunc main() {\n\tctx := context.Background()\n\tg := genkit.Init(ctx,\n\t\tgenkit.WithPlugins(&googlegenai.GoogleAI{}),\n\t\tgenkit.WithDefaultModel(\"googleai/gemini-2.5-flash-preview-tts\"),\n\t)\n\n\tgenkit.DefineFlow(g, \"textToSpeechFlow\",\n\t\tfunc(ctx context.Context, input struct {\n\t\t\tText      string `json:\"text\"`\n\t\t\tVoiceName string `json:\"voiceName,omitempty\"`\n\t\t}) (string, error) {\n\t\t\tvoiceName := input.VoiceName\n\t\t\tif voiceName == \"\" {\n\t\t\t\tvoiceName = \"Algenib\"\n\t\t\t}\n\n\t\t\tresponse, err := genkit.Generate(ctx, g,\n\t\t\t\tai.WithPrompt(input.Text),\n\t\t\t\tai.WithConfig(&genai.GenerateContentConfig{\n\t\t\t\t\tResponseModalities: []string{\"AUDIO\"},\n\t\t\t\t\tSpeechConfig: &genai.SpeechConfig{\n\t\t\t\t\t\tVoiceConfig: &genai.VoiceConfig{\n\t\t\t\t\t\t\tPrebuiltVoiceConfig: &genai.PrebuiltVoiceConfig{\n\t\t\t\t\t\t\t\tVoiceName: voiceName,\n\t\t\t\t\t\t\t},\n\t\t\t\t\t\t},\n\t\t\t\t\t},\n\t\t\t\t}),\n\t\t\t)\n\t\t\tif err != nil {\n\t\t\t\treturn \"\", err\n\t\t\t}\n\n\t\t\treturn response.Text(), nil\n\t\t},\n\t)\n\n\t<-ctx.Done()\n}\n```\n\n### Image Generation\n\n```go\npackage main\n\nimport (\n\t\"context\"\n\n\t\"github.com/firebase/genkit/go/ai\"\n\t\"github.com/firebase/genkit/go/genkit\"\n\t\"github.com/firebase/genkit/go/plugins/googlegenai\"\n\t\"google.golang.org/genai\"\n)\n\nfunc main() {\n\tctx := context.Background()\n\tg := genkit.Init(ctx, genkit.WithPlugins(&googlegenai.VertexAI{}))\n\n\tgenkit.DefineFlow(g, \"imageGenerationFlow\",\n\t\tfunc(ctx context.Context, prompt string) ([]string, error) {\n\t\t\tresponse, err := genkit.Generate(ctx, g,\n\t\t\t\tai.WithModelName(\"vertexai/imagen-3.0-generate-001\"),\n\t\t\t\tai.WithPrompt(\"Generate an image of %s\", prompt),\n\t\t\t\tai.WithConfig(&genai.GenerateImagesConfig{\n\t\t\t\t\tNumberOfImages:    2,\n\t\t\t\t\tAspectRatio:       \"9:16\",\n\t\t\t\t\tSafetyFilterLevel: genai.SafetyFilterLevelBlockLowAndAbove,\n\t\t\t\t\tPersonGeneration:  genai.PersonGenerationAllowAll,\n\t\t\t\t\tLanguage:          genai.ImagePromptLanguageEn,\n\t\t\t\t\tAddWatermark:      true,\n\t\t\t\t\tOutputMIMEType:    \"image/jpeg\",\n\t\t\t\t}),\n\t\t\t)\n\t\t\tif err != nil {\n\t\t\t\treturn nil, err\n\t\t\t}\n\n\t\t\tvar images []string\n\t\t\tfor _, part := range response.Message.Content {\n\t\t\t\timages = append(images, part.Text)\n\t\t\t}\n\t\t\treturn images, nil\n\t\t},\n\t)\n\n\t<-ctx.Done()\n}\n```\n\n---\n\n## Running and Inspecting Flows\n\n1.  **Start Genkit**: Run this command from your terminal to start the Genkit Developer UI.\n\n    ```bash\n    genkit start -- <command to run your code>\n    ```\n\n    For Go applications:\n\n    ```bash\n    # Running a Go application directly\n    genkit start -- go run main.go\n\n    # Running a compiled binary\n    genkit start -- ./my-genkit-app\n    ```\n\n    The command should output a URL for the Genkit Dev UI. Direct the user to visit this URL to run and inspect their Genkit app.\n\n## Suggested Models\n\nHere are suggested models to use for various task types. This is NOT an exhaustive list.\n\n### Advanced Text/Reasoning\n\n```\n| Plugin                                                     | Recommended Model                  |\n|------------------------------------------------------------|------------------------------------|\n| github.com/firebase/genkit/go/plugins/googlegenai         | gemini-2.5-pro                    |\n| github.com/firebase/genkit/go/plugins/compat_oai/openai   | gpt-4o                             |\n| github.com/firebase/genkit/go/plugins/compat_oai/deepseek | deepseek-reasoner                  |\n| github.com/firebase/genkit/go/plugins/compat_oai/xai      | grok-4                             |\n```\n\n### Fast Text/Chat\n\n```\n| Plugin                                                     | Recommended Model                  |\n|------------------------------------------------------------|------------------------------------|\n| github.com/firebase/genkit/go/plugins/googlegenai         | gemini-2.5-flash                  |\n| github.com/firebase/genkit/go/plugins/compat_oai/openai   | gpt-4o-mini                        |\n| github.com/firebase/genkit/go/plugins/compat_oai/deepseek | deepseek-chat                      |\n| github.com/firebase/genkit/go/plugins/compat_oai/xai      | grok-3-mini                        |\n```\n\n### Text-to-Speech\n\n```\n| Plugin                                                     | Recommended Model                  |\n|------------------------------------------------------------|------------------------------------|\n| github.com/firebase/genkit/go/plugins/googlegenai         | gemini-2.5-flash-preview-tts       |\n| github.com/firebase/genkit/go/plugins/compat_oai/openai   | gpt-4o-mini-tts                    |\n```\n\n### Image Generation\n\n```\n| Plugin                                                     | Recommended Model                  | Input Modalities  |\n|------------------------------------------------------------|------------------------------------|-------------------|\n| github.com/firebase/genkit/go/plugins/googlegenai         | gemini-2.5-flash-image-preview     | Text, Image       |\n| github.com/firebase/genkit/go/plugins/googlegenai         | imagen-4.0-generate-preview-06-06  | Text              |\n| github.com/firebase/genkit/go/plugins/compat_oai/openai   | gpt-image-1                        | Text              |\n```";
//# sourceMappingURL=go.js.map