import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
  CardDescription,
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Computer, HardDrive, Wrench, Box, PlusCircle, Search, FileText } from 'lucide-react';
import { inventory } from '@/lib/data';
import type { Asset } from '@/lib/data';
import Link from 'next/link';

function getStatusVariant(status: Asset['status']) {
  switch (status) {
    case 'Digunakan':
      return 'default';
    case 'Disimpan':
      return 'secondary';
    case 'Dalam Perawatan':
      return 'outline';
    case 'Dihapus':
      return 'destructive';
    default:
      return 'default';
  }
}

export default function DashboardPage() {
  const totalAssets = inventory.length;
  const inUse = inventory.filter((a) => a.status === 'Digunakan').length;
  const inMaintenance = inventory.filter(
    (a) => a.status === 'Dalam Perawatan'
  ).length;
  const inStorage = inventory.filter((a) => a.status === 'Disimpan').length;

  const recentActivity = inventory.slice(0, 5);

  return (
    <div className="flex flex-col gap-6">
      <h1 className="text-2xl font-bold font-headline">Dasbor</h1>
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Aset</CardTitle>
            <HardDrive className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{totalAssets}</div>
            <p className="text-xs text-muted-foreground">
              Semua komputer & periferal
            </p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Aset Digunakan</CardTitle>
            <Computer className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{inUse}</div>
            <p className="text-xs text-muted-foreground">Sedang aktif</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">
              Dalam Perawatan
            </CardTitle>
            <Wrench className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{inMaintenance}</div>
            <p className="text-xs text-muted-foreground">
              Sedang dalam perbaikan
            </p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Disimpan</CardTitle>
            <Box className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{inStorage}</div>
            <p className="text-xs text-muted-foreground">
              Tersedia untuk ditugaskan
            </p>
          </CardContent>
        </Card>
      </div>

      <div className="grid gap-6 md:grid-cols-5">
        <Card className="md:col-span-3">
          <CardHeader>
            <CardTitle>Aktivitas Terkini</CardTitle>
            <CardDescription>
              Aset yang baru ditambahkan atau diperbarui.
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Aset</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Tanggal</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {recentActivity.map((asset) => (
                  <TableRow key={asset.id}>
                    <TableCell>
                      <div className="font-medium">{asset.name}</div>
                      <div className="text-sm text-muted-foreground">
                        {asset.serialNumber}
                      </div>
                    </TableCell>
                    <TableCell>
                      <Badge variant={getStatusVariant(asset.status)}>
                        {asset.status}
                      </Badge>
                    </TableCell>
                    <TableCell>{asset.purchaseDate}</TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </CardContent>
        </Card>
        <Card className="md:col-span-2">
          <CardHeader>
            <CardTitle>Tindakan Cepat</CardTitle>
            <CardDescription>Lakukan tugas umum dengan cepat.</CardDescription>
          </CardHeader>
          <CardContent className="flex flex-col gap-4">
            <Link href="/inventory/add" passHref>
              <Button className="w-full justify-start">
                <PlusCircle className="mr-2 h-4 w-4" /> Tambah Aset Baru
              </Button>
            </Link>
            <Link href="/inventory" passHref>
              <Button variant="secondary" className="w-full justify-start">
                <Search className="mr-2 h-4 w-4" /> Cari Inventaris
              </Button>
            </Link>
            <Link href="/reports" passHref>
              <Button variant="secondary" className="w-full justify-start">
                <FileText className="mr-2 h-4 w-4" /> Buat Laporan
              </Button>
            </Link>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
