'use client';

import * as React from 'react';
import { useRouter } from 'next/navigation';
import { Button } from '@/components/ui/button';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import Link from 'next/link';
import { locations } from '@/lib/data';
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from '@/components/ui/popover';
import { cn } from '@/lib/utils';
import { Check, ChevronsUpDown } from 'lucide-react';
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from '@/components/ui/command';

export default function AddAssetPage() {
  const router = useRouter();
  const [open, setOpen] = React.useState(false);
  const [value, setValue] = React.useState('');

  const allLocations = React.useMemo(() => {
    const existingLocations = locations.map((loc) => loc.value);
    if (value && !existingLocations.includes(value)) {
      return [...locations, { value: value, label: value }];
    }
    return locations;
  }, [value]);

  const handleSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    const formData = new FormData(e.currentTarget);
    const newAsset = {
      id: crypto.randomUUID(),
      name: formData.get('name') as string,
      serialNumber: formData.get('serialNumber') as string,
      type: formData.get('type') as any,
      status: formData.get('status') as any,
      assignedTo: value,
      purchaseDate: formData.get('purchaseDate') as string,
      lastMaintenance: 'N/A',
      notes: formData.get('notes') as string,
    };

    // We use localStorage to pass the new asset to the inventory page
    localStorage.setItem('newAssets', JSON.stringify([newAsset]));

    router.push('/inventory');
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className="font-headline">Tambah Aset Baru</CardTitle>
        <CardDescription>
          Isi detail aset baru untuk menambahkannya ke inventaris.
        </CardDescription>
      </CardHeader>
      <CardContent>
        <form onSubmit={handleSubmit} className="grid gap-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-2">
              <Label htmlFor="name">Nama Aset</Label>
              <Input
                id="name"
                name="name"
                placeholder="e.g., Dell OptiPlex 7090"
                required
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="serialNumber">Nomor Seri</Label>
              <Input
                id="serialNumber"
                name="serialNumber"
                placeholder="e.g., SN-A1B2C3D4"
                required
              />
            </div>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-2">
              <Label htmlFor="type">Tipe Aset</Label>
              <Select name="type" required>
                <SelectTrigger id="type">
                  <SelectValue placeholder="Pilih tipe aset" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="Komputer">Komputer</SelectItem>
                  <SelectItem value="Periferal">Periferal</SelectItem>
                  <SelectItem value="Lainnya">Lainnya</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="status">Status</Label>
              <Select name="status" required>
                <SelectTrigger id="status">
                  <SelectValue placeholder="Pilih status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="Digunakan">Digunakan</SelectItem>
                  <SelectItem value="Disimpan">Disimpan</SelectItem>
                  <SelectItem value="Dalam Perawatan">
                    Dalam Perawatan
                  </SelectItem>
                  <SelectItem value="Dihapus">Dihapus</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-2">
              <Label htmlFor="assignedTo">Lokasi / Baris</Label>
              <Popover open={open} onOpenChange={setOpen}>
                <PopoverTrigger asChild>
                  <Button
                    variant="outline"
                    role="combobox"
                    aria-expanded={open}
                    className="w-full justify-between font-normal"
                  >
                    {value
                      ? allLocations.find((loc) => loc.value === value)?.label
                      : 'Pilih atau tambah lokasi...'}
                    <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                  </Button>
                </PopoverTrigger>
                <PopoverContent className="w-[--radix-popover-trigger-width] p-0">
                  <Command>
                    <CommandInput
                      placeholder="Cari atau tambah lokasi baru..."
                      onValueChange={setValue}
                    />
                    <CommandList>
                      <CommandEmpty>
                        Lokasi tidak ditemukan. Ketik untuk menambahkannya.
                      </CommandEmpty>
                      <CommandGroup>
                        {allLocations.map((loc) => (
                          <CommandItem
                            key={loc.value}
                            value={loc.value}
                            onSelect={(currentValue) => {
                              setValue(currentValue === value ? '' : currentValue);
                              setOpen(false);
                            }}
                          >
                            <Check
                              className={cn(
                                'mr-2 h-4 w-4',
                                value === loc.value
                                  ? 'opacity-100'
                                  : 'opacity-0'
                              )}
                            />
                            {loc.label}
                          </CommandItem>
                        ))}
                      </CommandGroup>
                    </CommandList>
                  </Command>
                </PopoverContent>
              </Popover>
            </div>
            <div className="space-y-2">
              <Label htmlFor="purchaseDate">Tanggal Pembelian</Label>
              <Input id="purchaseDate" name="purchaseDate" type="date" />
            </div>
          </div>
          <div className="space-y-2">
            <Label htmlFor="notes">Catatan</Label>
            <Textarea
              id="notes"
              name="notes"
              placeholder="e.g., Ditingkatkan ke SSD 1TB"
            />
          </div>
          <div className="flex justify-end gap-2">
            <Link href="/inventory" passHref>
              <Button variant="outline">Batal</Button>
            </Link>
            <Button type="submit">Simpan Aset</Button>
          </div>
        </form>
      </CardContent>
    </Card>
  );
}
