'use client';

import { useState, useMemo, useEffect, useRef } from 'react';
import Link from 'next/link';
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
  CardDescription,
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import {
  MoreHorizontal,
  PlusCircle,
  Search,
  QrCode,
  Upload,
  ScanLine,
} from 'lucide-react';
import { inventory as initialInventory } from '@/lib/data';
import type { Asset } from '@/lib/data';
import * as XLSX from 'xlsx';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { QrDialog } from '@/components/qr-dialog';
import { useToast } from '@/hooks/use-toast';

function getStatusVariant(status: Asset['status']) {
  switch (status) {
    case 'Digunakan':
      return 'default';
    case 'Disimpan':
      return 'secondary';
    case 'Dalam Perawatan':
      return 'outline';
    case 'Dihapus':
      return 'destructive';
    default:
      return 'default';
  }
}

export default function InventoryPage() {
  const [inventory, setInventory] = useState<Asset[]>(initialInventory);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedAsset, setSelectedAsset] = useState<Asset[] | null>(null);
  const [isQrDialogOpen, setIsQrDialogOpen] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const { toast } = useToast();

  useEffect(() => {
    const newAssetsString = localStorage.getItem('newAssets');
    if (newAssetsString) {
      try {
        const newAssets: Asset[] = JSON.parse(newAssetsString);
        if (Array.isArray(newAssets) && newAssets.length > 0) {
          setInventory((prev) => [...newAssets, ...prev]);
          setSelectedAsset(newAssets);
          setIsQrDialogOpen(true);
        }
      } catch (e) {
        console.error('Failed to parse new assets from localStorage', e);
      } finally {
        localStorage.removeItem('newAssets');
      }
    }

    const updatedAssetString = localStorage.getItem('updatedAsset');
    if (updatedAssetString) {
      try {
        const updatedAsset: Asset = JSON.parse(updatedAssetString);
        setInventory((prev) =>
          prev.map((asset) =>
            asset.id === updatedAsset.id ? updatedAsset : asset
          )
        );
      } catch (e) {
        console.error('Failed to parse updated asset from localStorage', e);
      } finally {
        localStorage.removeItem('updatedAsset');
      }
    }
  }, []);

  const filteredInventory = useMemo(() => {
    if (!searchTerm) return inventory;
    return inventory.filter(
      (asset) =>
        asset.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        asset.serialNumber.toLowerCase().includes(searchTerm.toLowerCase()) ||
        asset.assignedTo?.toLowerCase().includes(searchTerm.toLowerCase())
    );
  }, [searchTerm, inventory]);

  const handleShowQrCode = (asset: Asset) => {
    setSelectedAsset([asset]);
    setIsQrDialogOpen(true);
  };

  const handleImportClick = () => {
    fileInputRef.current?.click();
  };

  const handleFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = (e) => {
      try {
        const data = new Uint8Array(e.target?.result as ArrayBuffer);
        const workbook = XLSX.read(data, { type: 'array' });
        const sheetName = workbook.SheetNames[0];
        const worksheet = workbook.Sheets[sheetName];
        const json = XLSX.utils.sheet_to_json<any>(worksheet);

        const newAssets: Asset[] = json.map((row) => ({
          id: crypto.randomUUID(),
          name: row['Nama'] || 'Nama Tidak Ada',
          serialNumber: row['Nomor Seri'] || `SN-${Math.random().toString(36).substring(2, 10).toUpperCase()}`,
          type: row['Tipe'] || 'Lainnya',
          status: row['Status'] || 'Disimpan',
          assignedTo: row['Lokasi / Baris'],
          purchaseDate: row['Tanggal Pembelian'] || new Date().toISOString().split('T')[0],
          lastMaintenance: 'N/A',
          notes: row['Catatan'],
        }));

        setInventory((prev) => [...newAssets, ...prev]);
        setSelectedAsset(newAssets);
        setIsQrDialogOpen(true);
        toast({
          title: 'Impor Berhasil',
          description: `${newAssets.length} aset berhasil diimpor dari Excel.`,
        });
      } catch (error) {
        console.error('Error importing from Excel:', error);
        toast({
          title: 'Impor Gagal',
          description: 'Terjadi kesalahan saat memproses file Excel. Pastikan formatnya benar.',
          variant: 'destructive',
        });
      }
    };
    reader.readAsArrayBuffer(file);
    // Reset file input
    event.target.value = '';
  };

  return (
    <>
      <Card>
        <CardHeader>
          <div className="flex flex-col md:flex-row items-start md:items-center justify-between gap-4">
            <div>
              <CardTitle className="font-headline">Inventaris</CardTitle>
              <CardDescription>
                Kelola, lacak, dan impor semua aset lab Anda.
              </CardDescription>
            </div>
            <div className="flex items-center gap-2 w-full md:w-auto">
              <div className="relative flex-1 md:flex-initial">
                <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
                <Input
                  type="search"
                  placeholder="Cari berdasarkan nama, S/N..."
                  className="pl-8 sm:w-[200px] lg:w-[300px] w-full"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </div>
              <input
                type="file"
                ref={fileInputRef}
                className="hidden"
                onChange={handleFileChange}
                accept=".xlsx, .xls, .csv"
              />
              <Link href="/inventory/scan" passHref>
                <Button variant="outline">
                  <ScanLine className="mr-2 h-4 w-4" />
                  Pindai
                </Button>
              </Link>
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="outline">
                    <Upload className="mr-2 h-4 w-4" />
                    Impor
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end">
                  <DropdownMenuItem onClick={handleImportClick}>
                    Impor dari Excel
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>

              <Link href="/inventory/add" passHref>
                <Button>
                  <PlusCircle className="mr-2 h-4 w-4" />
                  Tambah Aset
                </Button>
              </Link>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Nama</TableHead>
                <TableHead className="hidden md:table-cell">
                  Nomor Seri
                </TableHead>
                <TableHead>Status</TableHead>
                <TableHead className="hidden md:table-cell">
                  Lokasi / Baris
                </TableHead>
                <TableHead className="hidden lg:table-cell">
                  Perawatan Terakhir
                </TableHead>
                <TableHead>
                  <span className="sr-only">Tindakan</span>
                </TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredInventory.length > 0 ? (
                filteredInventory.map((asset) => (
                  <TableRow key={asset.id}>
                    <TableCell className="font-medium">{asset.name}</TableCell>
                    <TableCell className="hidden md:table-cell">
                      {asset.serialNumber}
                    </TableCell>
                    <TableCell>
                      <Badge variant={getStatusVariant(asset.status)}>
                        {asset.status}
                      </Badge>
                    </TableCell>
                    <TableCell className="hidden md:table-cell">
                      {asset.assignedTo || 'N/A'}
                    </TableCell>
                    <TableCell className="hidden lg:table-cell">
                      {asset.lastMaintenance}
                    </TableCell>
                    <TableCell>
                      <DropdownMenu>
                        <DropdownMenuTrigger asChild>
                          <Button
                            aria-haspopup="true"
                            size="icon"
                            variant="ghost"
                          >
                            <MoreHorizontal className="h-4 w-4" />
                            <span className="sr-only">Buka menu</span>
                          </Button>
                        </DropdownMenuTrigger>
                        <DropdownMenuContent align="end">
                          <DropdownMenuLabel>Tindakan</DropdownMenuLabel>
                          <DropdownMenuItem
                            onSelect={() => handleShowQrCode(asset)}
                          >
                            <QrCode className="mr-2 h-4 w-4" />
                            Tampilkan Kode QR
                          </DropdownMenuItem>
                          <DropdownMenuSeparator />
                           <DropdownMenuItem asChild>
                            <Link href={`/inventory/edit/${asset.id}`}>Ubah</Link>
                          </DropdownMenuItem>
                          <DropdownMenuItem>Hapus</DropdownMenuItem>
                        </DropdownMenuContent>
                      </DropdownMenu>
                    </TableCell>
                  </TableRow>
                ))
              ) : (
                <TableRow>
                  <TableCell colSpan={6} className="h-24 text-center">
                    Tidak ada hasil yang ditemukan.
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </CardContent>
      </Card>
      <QrDialog
        assets={selectedAsset}
        isOpen={isQrDialogOpen}
        onClose={() => setIsQrDialogOpen(false)}
      />
    </>
  );
}
