'use client';

import { useEffect, useRef, useState } from 'react';
import { Html5QrcodeScanner, Html5Qrcode } from 'html5-qrcode';
import { useRouter } from 'next/navigation';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { CameraOff, X } from 'lucide-react';
import Link from 'next/link';

const qrboxFunction = (viewfinderWidth: number, viewfinderHeight: number) => {
  const minEdge = Math.min(viewfinderWidth, viewfinderHeight);
  const qrboxSize = Math.floor(minEdge * 0.7);
  return {
    width: qrboxSize,
    height: qrboxSize,
  };
};

export default function ScanPage() {
  const scannerRef = useRef<Html5Qrcode | null>(null);
  const router = useRouter();
  const [scanResult, setScanResult] = useState<string | null>(null);
  const [hasCameraPermission, setHasCameraPermission] = useState<boolean | null>(null);
  const [error, setError] = useState<string | null>(null);

  const onScanSuccess = (decodedText: string) => {
    setScanResult(decodedText);
    try {
      const qrData = JSON.parse(decodedText);
      if (qrData && qrData.id) {
        if (scannerRef.current?.isScanning) {
          scannerRef.current.stop();
        }
        router.push(`/inventory/edit/${qrData.id}`);
      } else {
        setError('Kode QR tidak valid. Format data tidak dikenali.');
      }
    } catch (e) {
      setError('Gagal memproses kode QR. Pastikan Anda memindai kode yang benar.');
    }
  };

  const onScanFailure = (errorMessage: string) => {
    // Abaikan kesalahan umum yang tidak relevan bagi pengguna
    if (errorMessage.includes('No QR code found')) {
      return;
    }
    setError(errorMessage);
  };
  
  useEffect(() => {
    const startScanner = async () => {
      try {
        const devices = await Html5Qrcode.getCameras();
        if (devices && devices.length) {
          setHasCameraPermission(true);
          const scanner = new Html5Qrcode('qr-reader');
          scannerRef.current = scanner;
          scanner.start(
            { facingMode: 'environment' },
            {
              fps: 10,
              qrbox: qrboxFunction,
              aspectRatio: 1,
            },
            onScanSuccess,
            onScanFailure
          );
        } else {
          setHasCameraPermission(false);
          setError('Tidak ada kamera yang ditemukan di perangkat ini.');
        }
      } catch (err) {
        setHasCameraPermission(false);
        if (typeof err === 'string') {
          setError(err);
        } else if (err instanceof Error) {
          if (err.name === 'NotAllowedError') {
             setError('Akses kamera ditolak. Izinkan akses kamera di pengaturan browser Anda untuk menggunakan fitur ini.');
          } else {
             setError(`Terjadi kesalahan saat memulai kamera: ${err.message}`);
          }
        }
      }
    };
    
    startScanner();

    return () => {
      if (scannerRef.current && scannerRef.current.isScanning) {
         scannerRef.current.stop().catch(console.error);
      }
    };
  }, []);

  return (
    <Card className="max-w-xl mx-auto">
      <CardHeader>
        <CardTitle className="font-headline">Pindai Kode QR Aset</CardTitle>
        <CardDescription>
          Arahkan kamera ke kode QR pada aset untuk melihat dan mengedit detailnya.
        </CardDescription>
      </CardHeader>
      <CardContent className="space-y-4">
        <div id="qr-reader" className="w-full aspect-square rounded-md overflow-hidden bg-muted flex items-center justify-center">
            {hasCameraPermission === false && (
                <div className="text-center text-muted-foreground p-4">
                    <CameraOff className="h-12 w-12 mx-auto mb-2"/>
                    <p>Izin kamera diperlukan</p>
                </div>
            )}
        </div>
        
        {error && (
           <Alert variant="destructive">
            <X className="h-4 w-4" />
            <AlertTitle>Terjadi Kesalahan</AlertTitle>
            <AlertDescription>
                {error}
            </AlertDescription>
          </Alert>
        )}

        <Link href="/inventory" passHref>
          <Button variant="outline" className="w-full">
            Kembali ke Inventaris
          </Button>
        </Link>
      </CardContent>
    </Card>
  );
}
