'use client';

import { useState } from 'react';
import { generateInventoryReport } from '@/ai/flows/generate-inventory-report';
import { Button } from '@/components/ui/button';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import { inventory, pastInventory } from '@/lib/data';
import { Loader2, Wand2, FileText, Download } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';
import * as XLSX from 'xlsx';

export default function ReportsPage() {
  const [reportType, setReportType] = useState<'weekly' | 'monthly' | ''>('');
  const [report, setReport] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const { toast } = useToast();

  const handleGenerateReport = async () => {
    if (!reportType) {
      toast({
        title: 'Kesalahan',
        description: 'Silakan pilih jenis laporan.',
        variant: 'destructive',
      });
      return;
    }

    setIsLoading(true);
    setReport('');

    try {
      const result = await generateInventoryReport({
        reportType,
        inventoryData: JSON.stringify(inventory),
        pastInventoryData: JSON.stringify(pastInventory),
      });
      setReport(result.report);
    } catch (error) {
      toast({
        title: 'Kesalahan Membuat Laporan',
        description:
          'Terjadi kesalahan saat membuat laporan. Silakan coba lagi.',
        variant: 'destructive',
      });
      console.error('Error generating report:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleDownloadExcel = () => {
    const worksheet = XLSX.utils.json_to_sheet(inventory);
    const workbook = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(workbook, worksheet, 'Inventaris');
    XLSX.writeFile(workbook, `Laporan_Inventaris_${new Date().toISOString().split('T')[0]}.xlsx`);
  };

  return (
    <div className="grid gap-6 lg:grid-cols-3">
      <div className="lg:col-span-1">
        <Card>
          <CardHeader>
            <CardTitle className="font-headline">
              Buat Laporan Inventaris
            </CardTitle>
            <CardDescription>
              Gunakan AI untuk menyusun ringkasan atau unduh data mentah sebagai Excel.
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-2">
                <Select
                  onValueChange={(value) =>
                    setReportType(value as 'weekly' | 'monthly')
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Pilih jenis ringkasan AI" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="weekly">Ringkasan Mingguan</SelectItem>
                    <SelectItem value="monthly">Ringkasan Bulanan</SelectItem>
                  </SelectContent>
                </Select>
                <Button
                  onClick={handleGenerateReport}
                  disabled={isLoading || !reportType}
                  className="w-full"
                >
                  {isLoading ? (
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  ) : (
                    <Wand2 className="mr-2 h-4 w-4" />
                  )}
                  Buat Ringkasan AI
                </Button>
            </div>
            <div className="space-y-2">
              <Button
                onClick={handleDownloadExcel}
                variant="secondary"
                className="w-full"
              >
                <Download className="mr-2 h-4 w-4" />
                Unduh Laporan Excel
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="lg:col-span-2">
        <Card className="min-h-[400px]">
          <CardHeader>
            <CardTitle className="font-headline">Ringkasan Laporan AI</CardTitle>
            <CardDescription>
              Laporan ringkasan yang dihasilkan AI akan muncul di bawah.
            </CardDescription>
          </CardHeader>
          <CardContent>
            {isLoading && (
              <div className="flex items-center justify-center p-8 h-64">
                <Loader2 className="h-8 w-8 animate-spin text-primary" />
              </div>
            )}
            {report && (
              <div className="rounded-md border bg-muted/30 p-4 text-sm">
                <p style={{ whiteSpace: 'pre-wrap' }}>{report}</p>
              </div>
            )}
            {!isLoading && !report && (
              <div className="flex flex-col items-center justify-center text-center p-8 border-2 border-dashed rounded-lg h-64">
                <FileText className="h-12 w-12 text-muted-foreground" />
                <p className="mt-4 text-sm text-muted-foreground">
                  Pilih jenis laporan dan klik "Buat Ringkasan AI" untuk melihat hasilnya di sini.
                </p>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
