'use client';

import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from '@/components/ui/dialog';
import { QRCodeSVG } from 'qrcode.react';
import type { Asset } from '@/lib/data';
import { Button } from '@/components/ui/button';
import { Printer } from 'lucide-react';
import { ScrollArea } from './ui/scroll-area';

interface QrDialogProps {
  assets: Asset[] | null;
  isOpen: boolean;
  onClose: () => void;
}

export function QrDialog({ assets, isOpen, onClose }: QrDialogProps) {
  const handlePrint = () => {
    const printableArea = document.getElementById('printable-area');
    if (printableArea) {
      const parent = printableArea.parentElement;
      if (parent) {
        // Temporarily give the printable area its own ID to be targeted by print styles
        const originalParentId = parent.id;
        parent.id = 'printable-area-container';
        
        // Isolate the dialog content for printing
        const appRoot = document.body.children[0];
        const originalBodyChildren = Array.from(document.body.children);
        const printableContainer = document.querySelector('[role="dialog"]');
        if(printableContainer) {
          document.body.innerHTML = '';
          document.body.appendChild(printableContainer)
        }
       
        window.print();
        
        // Restore original body
        document.body.innerHTML = '';
        originalBodyChildren.forEach(child => document.body.appendChild(child));
        
        if (parent) {
           parent.id = originalParentId;
        }
      }
    }
  };

  const getDialogTitle = () => {
    if (!assets) return '';
    if (assets.length === 1) return assets[0].name;
    return `Cetak Label untuk ${assets.length} Aset`;
  };
  
  const getDialogDescription = () => {
    if (!assets) return '';
    if (assets.length === 1) return assets[0].serialNumber;
    return 'Aset baru telah ditambahkan. Cetak label QR di bawah ini.';
  }

  if (!assets || assets.length === 0) {
    return null;
  }

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="sm:max-w-4xl print:shadow-none print:border-none print:max-w-full">
        <div id="printable-area">
          <DialogHeader className="print:hidden">
            <DialogTitle className="font-headline">{getDialogTitle()}</DialogTitle>
            <DialogDescription>
             {getDialogDescription()}
            </DialogDescription>
          </DialogHeader>
          
          <ScrollArea className="max-h-[60vh] print:max-h-full print:overflow-visible">
            <div className="p-1 grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 print:grid-cols-4">
              {assets.map((asset) => {
                const qrValue = JSON.stringify({
                  id: asset.id,
                  name: asset.name,
                  serialNumber: asset.serialNumber,
                });
                return (
                  <div key={asset.id} className="p-2 border rounded-lg flex flex-col items-center justify-center break-inside-avoid">
                      <QRCodeSVG value={qrValue} size={80} className="mb-2" />
                      <div className="text-center">
                        <p className="font-bold text-xs leading-tight">{asset.name}</p>
                        <p className="text-xxs text-muted-foreground">{asset.serialNumber}</p>
                      </div>
                  </div>
                );
              })}
            </div>
          </ScrollArea>
          
          <DialogDescription className="text-center text-xs text-muted-foreground pt-4 print:hidden">
            Pindai kode QR ini untuk mendapatkan detail aset. Sesuaikan skala cetak di browser Anda jika perlu.
          </DialogDescription>
        </div>
        <DialogFooter className="print:hidden">
          <Button variant="outline" onClick={onClose}>Batal</Button>
          <Button onClick={handlePrint}>
            <Printer className="mr-2 h-4 w-4" />
            Cetak Semua Label
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
